<?php
/**
 * BuddyPress Moderation Tools:- User UI Helper.
 *
 * @package    BuddyPress Moderation Tools
 * @subpackage User
 * @copyright  Copyright (c) 2018, Brajesh Singh
 * @license    https://www.gnu.org/licenses/gpl.html GNU Public License
 * @author     Brajesh Singh
 * @since      1.0.0
 */

namespace PressThemes\ModerationTools\Reportable_Types\User\Handlers;

// Exit if files accessed directly over web.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class User_UI_Helper
 *
 * Buttons & Messages management.
 */
class User_UI_Helper {

	/**
	 * Boot this class.
	 */
	public static function boot() {
		$self = new self();
		$self->setup();
	}

	/**
	 * Setup hooks callbacks
	 */
	public function setup() {

		add_filter( 'login_message', array( $this, 'login_blocked_message' ) );

		// filter body.
		// we may want to modify it later.
		if ( is_user_logged_in() && is_super_admin() ) {
			add_filter( 'body_class', array( $this, 'add_body_class' ) );
			add_filter( 'bp_get_member_class', array( $this, 'add_list_class' ) );
		}

		// Add Buttons.
		add_action( 'bp_member_header_actions', array( $this, 'single_member_button' ), 1001 );
		add_action( 'bp_directory_members_actions', array( $this, 'member_loop_button' ), 1001 );
	}

	/**
	 * Show notice to the blocked user on the login page.
	 *
	 * @param string $message message to be shown.
	 *
	 * @return string
	 */
	public function login_blocked_message( $message ) {
		// Show the error message if it seems to be a disabled user.
		$disabled_message = bpmts_get_option( 'user_disable_login_message' );

		if ( $disabled_message && ! empty( $_GET['bpmts_blocked'] ) ) {
			$message .= '<div id="login_error">' . $disabled_message . '</div>';
		}

		return $message;
	}

	/**
	 * Add appropriate class to body.
	 *
	 * @param array $classes css classes.
	 *
	 * @return array
	 */
	public function add_body_class( $classes ) {

		if ( ! function_exists( 'buddypress' ) ) {
			return $classes;
		}

		if ( bp_is_user() ) {
			$classes[] = bpmts_is_user_hidden( bp_displayed_user_id() ) ? 'bpmts-displayed-item-hidden bpmts-displayed-user-hidden' : 'bpmts-displayed-item-clean bpmts-displayed-user-clean';
		}

		if ( is_user_logged_in() ) {
			$classes[] = bpmts_is_user_hidden( bp_loggedin_user_id() ) ? 'bpmts-logged-user-hidden' : 'bpmts-logged-user-clean';
		}

		return $classes;
	}

	/**
	 * Add extra class to member list entry.
	 *
	 * @param array $classes classes.
	 *
	 * @return array
	 */
	public function add_list_class( $classes ) {
		$classes[] = bpmts_is_user_hidden( bp_get_member_user_id(), true ) ? 'bpmts-item-entry-hidden bpmts-member-entry-user-hidden' : 'bpmts-item-entry-clean bpmts-member-entry-user-clean';

		return $classes;
	}

	/**
	 * Add button to single member header.
	 */
	public function single_member_button() {

		if ( ! bpmts_user_can_report() ) {
			return;
		}

		$user_id = bp_displayed_user_id();

		if ( bpmts_is_me( $user_id ) ) {
			return;
		}

		bpmts_report_button( array(
			'item_id'    => $user_id,
			'item_type'  => 'user',
			'context'    => 'bp_user_header',
			'context_id' => bp_displayed_user_id(),
		) );
	}

	/**
	 * Member loop button.
	 */
	public function member_loop_button() {
		if ( ! bpmts_user_can_report() ) {
			return;
		}

		$logged_id = get_current_user_id();
		$user_id   = bp_get_member_user_id();

		if ( bpmts_is_me( $user_id ) ) {
			return;
		}

		bpmts_report_button( array(
			'item_id'    => $user_id,
			'item_type'  => 'user',
			'context'    => 'bp_users_list',
			'context_id' => $logged_id,
		) );

	}
}
