<?php
/**
 * Reportable Item Base class.
 *
 * @package    ReportAbuse
 * @subpackage Reportable
 * @copyright  Copyright (c) 2018, Brajesh Singh
 * @license    https://www.gnu.org/licenses/gpl.html GNU Public License
 * @author     Brajesh Singh
 * @since      1.0.0
 */

namespace PressThemes\ModerationTools\Reportable_Types;

use PressThemes\ModerationTools\Contracts\Reportable;
use PressThemes\ModerationTools\Models\Queue;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Reportable_Item
 *
 * @property-read string $type Reportable item type.
 * @property-read string $label Reportable item label.
 * @property-read int    $active is enabled.
 */
abstract class Reportable_Item_Type implements Reportable {

	/**
	 * Reportable Item type. Unique identifier for the reportable content type.
	 *
	 * @var string
	 */
	private $type;

	/**
	 * Label for the reportable item.
	 *
	 * @var string
	 */
	private $label;

	/**
	 * Is enabled?
	 *
	 * @var int
	 */
	private $active = 1;

	/**
	 * Guarded properties.
	 *
	 * @var array
	 */
	protected $guarded = array();

	/**
	 * Route constructor.
	 *
	 * @param string $type item type.
	 * @param string $label Item type label.
	 */
	public function __construct( $type, $label ) {
		$this->type  = trim( $type );
		$this->label = $label;

		// set state based on preference.
		$enabled_types = bpmts_get_option( 'reportable_items', array() );

		if ( ! in_array( $this->type, $enabled_types, true ) ) {
			$this->active = 0;
		}
	}

	/**
	 * Get a property.
	 *
	 * @param string $name property name.
	 *
	 * @return null|mixed
	 */
	public function __get( $name ) {

		if ( in_array( $name, $this->guarded, true ) || ! property_exists( $this, $name ) ) {
			return null; // or should we throw some exception.
		}

		return $this->{$name};
	}

	/**
	 * Get Post Permalink.
	 *
	 * @param int    $item_id post id.
	 * @param string $context ignore.
	 * @param int    $context_id ignore.
	 *
	 * @return bool
	 */
	public function report( $item_id, $context, $context_id ) {
	}

	/**
	 * Delete an item of this type.
	 *
	 * @param int $item_id item id.
	 */
	abstract public function delete( $item_id );

	/**
	 * Boot this type.
	 */
	public function boot() {
	}

	/**
	 * Setup the filters etc.
	 */
	public function setup() {
	}

	/**
	 * Add admin options.
	 *
	 * @param \Press_Themes\PT_Settings\Page $page page object.
	 */
	public function add_admin_settings( $page ) {

	}

	/**
	 * Get threshold to hide the item.
	 *
	 * @return int
	 */
	public function get_threshold() {
		return 10;
	}

	/**
	 * Get the item object.
	 *
	 * @param int $item_id user id.
	 *
	 * @return false|mixed
	 */
	abstract public function get_item( $item_id );

	/**
	 * Get the thumbnail
	 *
	 * @param int    $item_id item id.
	 * @param string $size thumb|full.
	 */
	public function get_item_thumbnail( $item_id, $size ) {

	}

	/**
	 * Get item title.
	 *
	 * @param int $item_id item id.
	 *
	 * @return string
	 */
	abstract public function get_item_title( $item_id );

	/**
	 * Get item description.
	 *
	 * @param int $item_id item id.
	 *
	 * @return string
	 */
	abstract public function get_item_description( $item_id );

	/**
	 * Item display card.
	 *
	 * @param int $item_id item id.
	 *
	 * @return string
	 */
	public function get_item_display_card( $item_id ) {
		$queue = Queue::first( array( 'item_id' => $item_id, 'item_type' => $this->type ) );

		$card = "<div class='bpmts-item-card bpmts-user-card bpmts-clearfix'>";
		$card .= "<div class='bpmts-item-thumbnail bpmts-type-{$this->type}-avatar'>{$this->get_item_thumbnail( $item_id, 'thumb' )}</div>";
		$card .= "<div class='bpmts-item-details'>";
		$card .= "<div class='bpmts-item-title bpmts-type-{$this->type}-title'>{$this->get_item_title( $item_id )}</div>";
		$card .= "<div class='bpmts-item-description bpmts-type-{$this->type}-description'>{$this->get_item_description( $item_id )}</div>";

		$links = array();
		$queue_id = is_null( $queue )? 0 : $queue->id;
		$is_hidden = is_null( $queue )? false : $queue->is_hidden;
		if ( $is_hidden ) {
			$restore_link = sprintf( '<a href="%s"  class="button bpmts-button-restore" title="%s">%s</a>', wp_nonce_url( bpmts_admin_get_queue_entry_url( array(
				'queue_id'  => $queue_id,
				'bpmts_action' => 'restore-item',
			) ), 'restore-item' ), __( 'Restore item', 'buddypress-moderation-tools' ),  __( 'Restore', 'buddypress-moderation-tools' ) );
		} else {
			$restore_link = sprintf( '<a href="%s"  class="button bpmts-button-clean" title="%s">%s</a>', wp_nonce_url( bpmts_admin_get_queue_entry_url( array(
				'queue_id'  => $queue_id,
				'bpmts_action' => 'clean-item',
			) ), 'clean-item' ),__( 'Remove the item from queue.', 'buddypress-moderation-tools' ),  __( 'Mark Clean', 'buddypress-moderation-tools' ) );
		}

		if ( $queue ) {
			$links = array(
				'restore' => $restore_link,
				'delete'  => sprintf( '<a href="%s" class="button bpmts-button-delete" title="%s">%s</a>', wp_nonce_url( bpmts_admin_get_queue_entry_url( array(
					'queue_id'     => $queue_id,
					'bpmts_action' => 'delete-item',
				) ), 'delete-item' ), __( 'Be careful. The item will be deleted permanently.', 'buddypress-moderation-tools' ), __( 'Delete', 'buddypress-moderation-tools' ) ),
			);

		}

		$card .= "<div class='bpmts-item-actions'><ul>";

		foreach ( $links as $link ) {
			$card .= "<li>$link</li>";
		}

		$card .= '</ul></div>';

		$card .= '</div>'; // end of details.


		$card .= "<div class='bpmts-item-footer'>";

		if ( $queue ) {
			$card .= '<ul>';

			$card .= '<li>' . sprintf( __( 'Type: <span> %s</span>', 'buddypress-moderation-tools' ), $this->label ) . '</li>';
			$card .= '<li>' . sprintf( __( 'Total Reports: <span> %d</span>', 'buddypress-moderation-tools' ), $queue->total_reports_count ) . '</li>';
			$card .= '<li>' . sprintf( __( 'New Reports: <span> %d</span>', 'buddypress-moderation-tools' ), $queue->current_reports_count ) . '</li>';

			if ( $queue->is_hidden ) {
				$card .= '<li>' . __( 'State: <span>Hidden</span>', 'buddypress-moderation-tools' ) . '</li>';
			} else {
				$card .= '<li>' . __( 'State: <span>Visible</span>', 'buddypress-moderation-tools' ) . '</li>';
			}

			$card .= '<li>' . sprintf( __( 'First reported on: <span> %s</span>', 'buddypress-moderation-tools' ), mysql2date( 'F j, Y',  $queue->created_at ) ) . '</li>';
			$card .= '<li>' . sprintf( __( 'Last reported on: <span> %s</span>', 'buddypress-moderation-tools' ), mysql2date( 'F j, Y',  $queue->updated_at ) ) . '</li>';

			$card .= '</ul>';
		}

		$card .= '</div>';
		$card .= '</div>';

		return $card;
	}

	/**
	 * Get item permalink.
	 *
	 * @param int $item_id item id.
	 *
	 * @return string
	 */
	public function get_item_permalink( $item_id ) {
		return '';
	}

	/**
	 * Get item edit url.
	 *
	 * @param int $item_id item id.
	 *
	 * @return string
	 */
	public function get_item_edit_url( $item_id ) {
		return '';
	}

	/**
	 * Get item delete url.
	 *
	 * @param int $item_id item id.
	 *
	 * @return string
	 */
	public function get_item_delete_url( $item_id ) {
		return '';
	}

}
