<?php
/**
 * BuddyPress Group Reportable support.
 *
 * @package    ModerationTools
 * @subpackage Reportable
 * @copyright  Copyright (c) 2018, Brajesh Singh
 * @license    https://www.gnu.org/licenses/gpl.html GNU Public License
 * @author     Brajesh Singh
 * @since      1.0.0
 */

namespace PressThemes\ModerationTools\Reportable_Types\BuddyPress;

use PressThemes\ModerationTools\Reportable_Types\BuddyPress\Group\Filters\Group_Query_Filter;
use PressThemes\ModerationTools\Reportable_Types\BuddyPress\Group\Admin\Group_Settings_Panel;

use PressThemes\ModerationTools\Reportable_Types\BuddyPress\Group\Handlers\Group_Action_Handler;
use PressThemes\ModerationTools\Reportable_Types\BuddyPress\Group\Handlers\Group_UI_Helper;
use PressThemes\ModerationTools\Reportable_Types\Reportable_Item_Type;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Reportable group type
 */
class Reportable_Type_BP_Group extends Reportable_Item_Type {

	/**
	 * Reportable_group constructor.
	 */
	public function __construct() {
		parent::__construct( 'bp_group', __( 'BuddyPress Group', 'buddypress-moderation-tools' ) );
	}

	/**
	 * Boot this module.
	 */
	public function boot() {
		$path = dirname( __FILE__ );
		require_once $path . '/group/bpmts-group-functions.php';
	}

	/**
	 * Setup hooks. Called after boot.
	 */
	public function setup() {
		// do some setup.
		// Boot action handlers.
		Group_Action_Handler::boot();
		Group_Query_Filter::boot();
		// Boot UI Helper to add messages/buttons as needed.
		Group_UI_Helper::boot();
	}

	/**
	 * Add admin settings.
	 *
	 * @param \Press_Themes\PT_Settings\Page $page page object.
	 */
	public function add_admin_settings( $page ) {
		$panel = $page->add_panel( 'bp_group', _x( 'Group', 'Admin settings panel title', 'buddypress-moderation-tools' ) );
		Group_Settings_Panel::build( $panel );
	}

	/**
	 * Delete an item of this type.
	 *
	 * @param int $item_id item id.
	 */
	public function delete( $item_id ) {
		groups_delete_group( $item_id );
	}

	/**
	 * Check that group exists.
	 *
	 * @param int    $item_id post id.
	 * @param string $context ignore.
	 * @param int    $context_id ignore.
	 *
	 * @return bool
	 */
	public function is_valid( $item_id, $context, $context_id ) {
		$group = new \BP_Groups_Group( $item_id );
		return $group && $group->id;
	}

	/**
	 * Get user threshold.
	 *
	 * @return int
	 */
	public function get_threshold() {
		return bpmts_get_option( 'bp_group_threshold', 5 );
	}

	/**
	 * Get the item object.
	 *
	 * @param int $item_id group id.
	 *
	 * @return false|\BP_Groups_Group
	 */
	public function get_item( $item_id ) {
		return groups_get_group( $item_id );
	}

	/**
	 * Get the thumbnail
	 *
	 * @param int    $item_id item id.
	 * @param string $size thumb|full.
	 *
	 * @return string
	 */
	public function get_item_thumbnail( $item_id, $size ) {
		return bp_core_fetch_avatar( array( 'item_id' => $item_id, 'size' => $size, 'object' => 'group' ) );
	}

	/**
	 * Get item title.
	 *
	 * @param int $item_id item id.
	 *
	 * @return string
	 */
	public function get_item_title( $item_id ) {
		$group = $this->get_item( $item_id );

		return $group? $group->name : __( 'Invalid Group.', 'buddypress-moderation-tools' );
	}

	/**
	 * Get item description.
	 *
	 * @param int $item_id item id.
	 *
	 * @return string
	 */
	public function get_item_description( $item_id ) {
		return bp_get_group_description_excerpt( $this->get_item( $item_id ) );
	}

	/**
	 * Get Post Permalink.
	 *
	 * @param int $item_id user id.
	 *
	 * @return false|string
	 */
	public function get_item_permalink( $item_id ) {
		return bp_get_group_permalink( $this->get_item( $item_id ) );
	}

	/**
	 * Get item edit url.
	 *
	 * @param int $item_id item id.
	 *
	 * @return string
	 */
	public function get_item_edit_url( $item_id ) {
		return bp_get_group_admin_permalink( $this->get_item( $item_id ) );
	}

	/**
	 * Get item delete url.
	 *
	 * @param int $item_id item id.
	 *
	 * @return string
	 */
	public function get_item_delete_url( $item_id ) {
		return bp_get_group_admin_permalink( $this->get_item( $item_id ) );
	}
}
