<?php
/**
 * BuddyPress Group Reportable support.
 *
 * @package    BuddyPress Moderation Tools
 * @subpackage PressThemes\ModerationTools\Reportable_Types\BBPress
 * @copyright  Copyright (c) 2018, Brajesh Singh
 * @license    https://www.gnu.org/licenses/gpl.html GNU Public License
 * @author     Brajesh Singh, Ravi Sharma
 * @since      1.0.0
 */

namespace PressThemes\ModerationTools\Reportable_Types\BBPress;

use PressThemes\ModerationTools\Reportable_Types\BBPress\Topic\Filters\Topic_Query_Filter;
use PressThemes\ModerationTools\Reportable_Types\BBPress\Topic\Admin\Topic_Settings_Panel;
use PressThemes\ModerationTools\Reportable_Types\BBPress\Topic\Handlers\Topic_Action_Handler;
use PressThemes\ModerationTools\Reportable_Types\BBPress\Topic\Handlers\Topic_UI_Helper;
use PressThemes\ModerationTools\Reportable_Types\Reportable_Item_Type;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Reportable_Type_BBP_Topic Reportable reply
 */
class Reportable_Type_BBP_Topic extends Reportable_Item_Type {

	/**
	 * Reportable_Type_BBP_Topic constructor.
	 */
	public function __construct() {
		parent::__construct( 'topic', __( 'bbPress Topic', 'buddypress-moderation-tools' ) );
	}

	/**
	 * Boot this module.
	 */
	public function boot() {
	}

	/**
	 * Setup hooks. Called after boot.
	 */
	public function setup() {
		// do some setup.
		// Boot action handlers.
		Topic_Action_Handler::boot();
		Topic_Query_Filter::boot();
		// Boot UI Helper to add messages/buttons as needed.
		Topic_UI_Helper::boot();
	}

	/**
	 * Add admin settings.
	 *
	 * @param \Press_Themes\PT_Settings\Page $page page object.
	 */
	public function add_admin_settings( $page ) {
		$panel = $page->add_panel( 'topic', _x( 'bbPress Topic', 'Admin settings panel title', 'buddypress-moderation-tools' ) );

		Topic_Settings_Panel::build( $panel );
	}

	/**
	 * Delete an item of this type.
	 *
	 * @param int $item_id item id.
	 */
	public function delete( $item_id ) {
		bbp_delete_topic( $item_id );
	}

	/**
	 * Check that group exists.
	 *
	 * @param int    $item_id post id.
	 * @param string $context ignore.
	 * @param int    $context_id ignore.
	 *
	 * @return bool
	 */
	public function is_valid( $item_id, $context, $context_id ) {
		$topic = bbp_get_topic( $item_id );

		if ( empty( $topic ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Get user threshold.
	 *
	 * @return int
	 */
	public function get_threshold() {
		return bpmts_get_option( 'bbp_topic_threshold', 5 );
	}

	/**
	 * Get the item object.
	 *
	 * @param int $item_id Reply id.
	 *
	 * @return mixed Null if error or reply (in specified form) if success
	 */
	public function get_item( $item_id ) {
		return bbp_get_topic( $item_id );
	}

	/**
	 * Get the thumbnail
	 *
	 * @param int    $item_id item id.
	 * @param string $size thumb|full.
	 *
	 * @return string
	 */
	public function get_item_thumbnail( $item_id, $size ) {
		return get_avatar( bbp_get_topic_author_id( $item_id ), $size );
	}

	/**
	 * Get item title.
	 *
	 * @param int $item_id item id.
	 *
	 * @return string
	 */
	public function get_item_title( $item_id ) {
		return bbp_get_topic_title( $item_id );
	}

	/**
	 * Get item description.
	 *
	 * @param int $item_id item id.
	 *
	 * @return string
	 */
	public function get_item_description( $item_id ) {
		$reply = $this->get_item( $item_id );
		return ! empty( $reply ) ? $reply->post_excerpt : '';
	}

	/**
	 * Get Reply Permalink.
	 *
	 * @param int $item_id user id.
	 *
	 * @return false|string
	 */
	public function get_item_permalink( $item_id ) {
		return bbp_get_topic_permalink( $item_id );
	}

	/**
	 * Get item edit url.
	 *
	 * @param int $item_id item id.
	 *
	 * @return string
	 */
	public function get_item_edit_url( $item_id ) {
		return get_edit_post_link( $item_id );
	}

	/**
	 * Get item delete url.
	 *
	 * @param int $item_id item id.
	 *
	 * @return string
	 */
	public function get_item_delete_url( $item_id ) {
		return get_delete_post_link( $item_id );
	}
}
