<?php
/**
 * Model class for moderation log table
 *
 * @package buddypress-moderation-tools
 */

namespace PressThemes\ModerationTools\Models;

use PressThemes\ModerationTools\Schema\Schema;

/**
 * Class Category
 *
 * @property-read int    $id       Row id.
 * @property-read string $name    Queue id.
 * @property-read int    $is_active is active.
 * @property-read int    $position sort order.
 */
class Category extends Model {

	/**
	 * Do not store timestamp.
	 *
	 * @var bool
	 */
	protected $timestamps = false;

	/**
	 * Get the table name.
	 *
	 * @return string
	 */
	public static function table() {
		return Schema::table( 'category' );
	}

	/**
	 * Get table schema.
	 *
	 * @return array
	 */
	public static function schema() {

		return array(
			'id'       => 'integer',
			'name'     => 'string',
			'position' => 'integer',
			'is_active'   => 'integer',
		);
	}

	/**
	 * Get all active categories.
	 *
	 * @return bool|mixed|static[]
	 */
	public static function get_active() {

		$cats = wp_cache_get( 'active_categories', 'bpmts' );
		if ( false !== $cats ) {
			return $cats;
		}

		$cats = static::get(
			array(
				'is_active' => 1,
				'per_page'  => 0,
			)
		);

		$ncats = array();
		foreach ( $cats as $cat ) {
			$ncats[ $cat->id ] = $cat;
		}
		wp_cache_set( 'active_categories', $ncats, 'bpmts' );

		return $ncats;
	}

	/**
	 * Get the category name from id.
	 *
	 * @param int $id category id.
	 *
	 * @return string
	 */
	public static function name( $id ) {
		if ( ! $id ) {
			return __( 'Not Specified', 'buddypress-moderation-tools' );
		}

		$active = self::get_active();
		if ( isset( $active[ $id ] ) && $active[ $id ] ) {
			$cat = $active[ $id ];
		} else {
			$cat = self::find( $id );
		}

		if ( ! $cat ) {
			return __( 'Unavailable', 'buddypress-moderation-tools' );
		}

		return $cat->name;
	}
}
