<?php
/**
 * Non Persistent cahe. Valid for single request.
 *
 * @package    BuddyPress Moderation Tools
 * @subpackage Handlers
 * @copyright  Copyright (c) 2018, Brajesh Singh
 * @license    https://www.gnu.org/licenses/gpl.html GNU Public License
 * @author     Brajesh Singh
 * @since      1.0.0
 */

namespace PressThemes\ModerationTools\Handlers;

// Do not allow direct access over web.
defined( 'ABSPATH' ) || exit;

/**
 * In Memory, Non Persistent caching helper.
 */
class NP_Cache {

	/**
	 * Item store.
	 *
	 * @var array
	 */
	private $store = array();

	/**
	 * Add to cache only if the value does not exists already.
	 *
	 * @param string $key key.
	 * @param mixed  $value value.
	 * @param string $group groups.
	 *
	 * @return bool
	 */
	public function add( $key, $value, $group = '' ) {
		if( $this->has( $key, $group ) ) {
			return false;
		}

		return $this->set( $key, $value, $group );
	}

	/**
	 * Set data. Overrides existing value.
	 *
	 * @param string $key key.
	 * @param mixed  $value value.
	 * @param string $group groups.
	 *
	 * @return bool
	 */
	public function set( $key, $value, $group = '' ) {
		$key                 = $this->key( $key, $group );
		$this->store[ $key ] = $value;

		return true;
	}

	/**
	 * Get from cache.
	 *
	 * @param string $key key.
	 * @param string $group groups.
	 *
	 * @return mixed|null
	 */
	public function get( $key, $group = '' ) {
		$key = $this->key( $key, $group );

		return isset( $this->store[ $key ] ) ?$this->store[ $key ] : null;
	}
	/**
	 * Check if the key exists.
	 *
	 * @param string $key key.
	 * @param string $group groups.
	 *
	 * @return bool
	 */
	public function has( $key, $group = '' ) {
		$key = $this->key( $key, $group );

		return isset( $this->store[ $key ] );
	}

	/**
	 * Delete a thing from cache.
	 *
	 * @param string $key key.
	 * @param string $group groups.
	 */
	public function delete( $key, $group = '' ) {
		$key = $this->key( $key, $group );
		unset( $this->store[ $key ] );
	}

	/**
	 * Get key
	 *
	 * @param string $key key.
	 * @param string $group cache group.
	 *
	 * @return string
	 */
	private function key( $key, $group = '' ) {
		return empty( $group ) ? $key : "{$key}-{$group}";
	}
}