<?php
/**
 * Plugin core functions file
 *
 * @package buddypress-moderation-tools
 */

use PressThemes\ModerationTools\Models\Log;
use PressThemes\ModerationTools\Models\Queue;
use PressThemes\ModerationTools\Models\Report;

// Do not allow direct access over web.
defined( 'ABSPATH' ) || exit;

/**
 * Get default settings
 *
 * @return mixed
 */
function bpmts_get_default_settings() {

	$defaults = array(
		'reportable_items'    => array(
			'post'        => 'post',
			'user'        => 'user',
			'bp_group'    => 'bp_group',
			'bp_activity' => 'bp_activity',
			'bp_acomment' => 'bp_acomment',
			'bp_message'  => 'bp_message',
		),
		'who_can_report'      => 'logged',
		'moderation_criteria' => 'roles',

		'keep_log'                         => 1,
		'form_fields'                      => array(
			'subject'  => 'subject',
			'message'  => 'message',
			'category' => 'category',
			'name'     => 'name',
			'email'    => 'email',

		),
		'required_form_fields'             => array(
			'subject'  => 'subject',
			'message'  => 'message',
			'category' => 'category',
			'name'     => 'name',
			'email'    => 'email',

		),

		// User.
		'user_auto_hide'                   => 1,
		'user_threshold'                   => 5,
		'user_disable_login'               => 1,
		'user_disable_login_message'       => __( 'Your account is under moderation. Please contact the support for further assistance.', 'buddypress-moderation-tools' ),
		'user_hidden_user_message'         => __( 'Your account is under moderation. Please contact the support for further assistance.', 'buddypress-moderation-tools' ),
		'user_hide_type'                   => array(
			'all' => 'all',
		),
		// 'members'.
		'user_logout_on_hide'              => 1,
		'user_disable_private_message'     => 1,
		'user_disable_activity_posting'    => 1,
		'user_disable_activity_commenting' => 1,

		'user_notify_admin_on_report'   => 1,
		'user_admin_new_report_subject' => __( 'New: User reported', 'buddypress-moderation-tools' ),
		'user_admin_new_report_message' => '',


		'user_notify_admin_on_hide' => 1,
		'user_admin_hidden_subject' => __( 'New: User Hidden', 'buddypress-moderation-tools' ),
		'user_admin_hidden_message' => '',

		'user_notify_user_on_hide' => 1,
		'user_user_hidden_subject' => __( 'Your account is under moderation', 'buddypress-moderation-tools' ),
		'user_user_hidden_message' => '',

		'user_notify_user_on_restore' => 1,
		'user_user_restored_subject'  => __( 'Your account is restored', 'buddypress-moderation-tools' ),
		'user_user_restored_message'  => '',

		'user_notify_user_on_delete' => 1,
		'user_user_deleted_subject'  => __( 'Your account has been removed', 'buddypress-moderation-tools' ),
		'user_user_deleted_message'  => '',

		// Posts.
		'post_auto_hide'             => 0,
		'post_threshold'             => 5,
		'post_hide_type'             => array(
			'all' => 'all',
		),
		'post_notify_admin_on_report' => 1,
		'post_admin_reported_subject' => __( 'New: Post reported', 'buddypress-moderation-tools' ),
		'post_admin_reported_message' => '',

		'post_notify_admin_on_hide' => 1,
		'post_admin_hidden_subject' => __( 'New: post hidden', 'buddypress-moderation-tools' ),
		'post_admin_hidden_message' => '',

		'post_notify_author_on_hide' => 1,
		'post_author_hidden_subject' => __( 'Your post is under moderation', 'buddypress-moderation-tools' ),
		'post_author_hidden_message' => '',

		'post_notify_author_on_restore' => 1,
		'post_author_restored_subject'  => __( 'Your post is restored', 'buddypress-moderation-tools' ),
		'post_author_restored_message'  => '',

		'post_notify_author_on_delete' => 1,
		'post_author_deleted_subject'  => __( 'Your post has been removed.', 'buddypress-moderation-tools' ),
		'post_author_deleted_message'  => '',

		// Groups.
		'bp_group_auto_hide'           => 1,
		'bp_group_hide_type'           => array( 'all' => 'all' ),
		'bp_group_threshold'           => 5,
		'bp_group_admin_can_moderate'  => 1,

		'bp_group_notify_admin_on_report' => 1,
		'bp_group_admin_reported_subject' => __( 'New: Group reported', 'buddypress-moderation-tools' ),
		'bp_group_admin_reported_message' => '',


		'bp_group_notify_admin_on_hide' => 1,
		'bp_group_admin_hidden_subject' => __( 'New: Group hidden', 'buddypress-moderation-tools' ),
		'bp_group_admin_hidden_message' => '',

		'bp_group_notify_gadmin_on_hide' => 1,
		'bp_group_gadmin_hidden_subject' => __( 'Your group is under moderation', 'buddypress-moderation-tools' ),
		'bp_group_gadmin_hidden_message' => '',

		'bp_group_notify_gadmin_on_restore' => 1,
		'bp_group_gadmin_restored_subject'  => __( 'Your group is restored', 'buddypress-moderation-tools' ),
		'bp_group_gadmin_restored_message'  => '',

		'bp_group_notify_gadmin_on_delete' => 1,
		'bp_group_gadmin_deleted_subject'  => __( 'Your group has been removed.', 'buddypress-moderation-tools' ),
		'bp_group_gadmin_deleted_message'  => '',

		'bp_activity_auto_hide' => 1,
		'bp_activity_threshold' => 5,
		'bp_activity_hide_type' => array( 'all' => 'all' ),

		'bp_activity_notify_admin_on_report' => 1,
		'bp_activity_admin_reported_subject' => __( 'New: activity reported', 'buddypress-moderation-tools' ),
		'bp_activity_admin_reported_message' => '',


		'bp_activity_notify_admin_on_hide' => 1,
		'bp_activity_admin_hidden_subject' => __( 'New: activity hidden', 'buddypress-moderation-tools' ),
		'bp_activity_admin_hidden_message' => '',

		'bp_activity_notify_user_on_hide' => 1,
		'bp_activity_user_hidden_subject' => __( 'Your activity is under moderation', 'buddypress-moderation-tools' ),
		'bp_activity_user_hidden_message' => '',

		'bp_activity_notify_user_on_restore' => 1,
		'bp_activity_user_restored_subject'  => __( 'Your activity is restored', 'buddypress-moderation-tools' ),
		'bp_activity_user_restored_message'  => '',

		'bp_activity_notify_user_on_delete'  => 1,
		'bp_activity_user_deleted_subject'   => __( 'Your activity has been removed.', 'buddypress-moderation-tools' ),
		'bp_activity_user_deleted_message'   => '',

		// Activity Comments.
		'bp_acomment_auto_hide'              => 1,
		'bp_acomment_threshold'              => 5,
		'bp_acomment_hide_type'              => array( 'all' => 'all' ),
		'bp_acomment_notify_admin_on_report' => 1,
		'bp_acomment_admin_reported_subject' => __( 'New: activity comment reported', 'buddypress-moderation-tools' ),
		'bp_acomment_admin_reported_message' => '',


		'bp_acomment_notify_admin_on_hide' => 1,
		'bp_acomment_admin_hidden_subject' => __( 'New: activity comment hidden', 'buddypress-moderation-tools' ),
		'bp_acomment_admin_hidden_message' => '',

		'bp_acomment_notify_user_on_hide' => 1,
		'bp_acomment_user_hidden_subject' => __( 'Your comment is under moderation', 'buddypress-moderation-tools' ),
		'bp_acomment_user_hidden_message' => '',

		'bp_acomment_notify_user_on_restore' => 1,
		'bp_acomment_user_restored_subject'  => __( 'Your comment is restored', 'buddypress-moderation-tools' ),
		'bp_acomment_user_restored_message'  => '',

		'bp_acomment_notify_user_on_delete' => 1,
		'bp_acomment_user_deleted_subject'  => __( 'Your comment has been removed.', 'buddypress-moderation-tools' ),
		'bp_acomment_user_deleted_message'  => '',

		'bbp_reply_auto_hide'             => 0,
		'bbp_reply_threshold'             => 5,
		'bbp_reply_hide_type'             => array(
			'all' => 'all',
		),
		'bbp_reply_notify_admin_on_report' => 1,
		'bbp_reply_admin_reported_subject' => __( 'New: reply reported', 'buddypress-moderation-tools' ),
		'bbp_reply_admin_reported_message' => '',

		'bbp_reply_notify_admin_on_hide' => 1,
		'bbp_reply_admin_hidden_subject' => __( 'New: reply hidden', 'buddypress-moderation-tools' ),
		'bbp_reply_admin_hidden_message' => '',

		'bbp_reply_notify_user_on_hide' => 1,
		'bbp_reply_user_hidden_subject' => __( 'Your reply is under moderation', 'buddypress-moderation-tools' ),
		'bbp_reply_user_hidden_message' => '',

		'bbp_reply_notify_user_on_restore' => 1,
		'bbp_reply_user_restored_subject'  => __( 'Your reply is restored', 'buddypress-moderation-tools' ),
		'bbp_reply_user_restored_message'  => '',

		'bbp_reply_notify_user_on_delete' => 1,
		'bbp_reply_user_deleted_subject'  => __( 'Your reply has been removed.', 'buddypress-moderation-tools' ),
		'bbp_reply_user_deleted_message'  => '',

		'bbp_forum_auto_hide'             => 0,
		'bbp_forum_threshold'             => 5,
		'bbp_forum_hide_type'             => array(
			'all' => 'all',
		),
		'bbp_forum_notify_admin_on_report' => 1,
		'bbp_forum_admin_reported_subject' => __( 'New: forum reported', 'buddypress-moderation-tools' ),
		'bbp_forum_admin_reported_message' => '',

		'bbp_forum_notify_admin_on_hide' => 1,
		'bbp_forum_admin_hidden_subject' => __( 'New: forum hidden', 'buddypress-moderation-tools' ),
		'bbp_forum_admin_hidden_message' => '',

		'bbp_forum_notify_user_on_hide' => 1,
		'bbp_forum_user_hidden_subject' => __( 'Your forum is under moderation', 'buddypress-moderation-tools' ),
		'bbp_forum_user_hidden_message' => '',

		'bbp_forum_notify_user_on_restore' => 1,
		'bbp_forum_user_restored_subject'  => __( 'Your forum is restored', 'buddypress-moderation-tools' ),
		'bbp_forum_user_restored_message'  => '',

		'bbp_forum_notify_user_on_delete' => 1,
		'bbp_forum_user_deleted_subject'  => __( 'Your forum has been removed.', 'buddypress-moderation-tools' ),
		'bbp_forum_user_deleted_message'  => '',

		'bbp_topic_auto_hide'             => 0,
		'bbp_topic_threshold'             => 5,
		'bbp_topic_hide_type'             => array(
			'all' => 'all',
		),
		'bbp_topic_notify_admin_on_report' => 1,
		'bbp_topic_admin_reported_subject' => __( 'New: topic reported', 'buddypress-moderation-tools' ),
		'bbp_topic_admin_reported_message' => '',

		'bbp_topic_notify_admin_on_hide' => 1,
		'bbp_topic_admin_hidden_subject' => __( 'New: topic hidden', 'buddypress-moderation-tools' ),
		'bbp_topic_admin_hidden_message' => '',

		'bbp_topic_notify_user_on_hide' => 1,
		'bbp_topic_user_hidden_subject' => __( 'Your topic is under moderation', 'buddypress-moderation-tools' ),
		'bbp_topic_user_hidden_message' => '',

		'bbp_topic_notify_user_on_restore' => 1,
		'bbp_topic_user_restored_subject'  => __( 'Your topic is restored', 'buddypress-moderation-tools' ),
		'bbp_topic_user_restored_message'  => '',

		'bbp_topic_notify_user_on_delete' => 1,
		'bbp_topic_user_deleted_subject'  => __( 'Your topic has been removed.', 'buddypress-moderation-tools' ),
		'bbp_topic_user_deleted_message'  => '',
	);

	// User.
	$defaults['user_admin_new_report_message'] = <<<'EOD'
Hello {user.displayname},

A new {item.type} with title {item.label} has been marked as abusive.

Reported Content: {report.content}

Reported By:   {report.reporter}
Reported Date: {report.date}
View Report:   {report.link}
EOD;

	$defaults['user_admin_hidden_message'] = <<<'EOD'
Hello {user.displayname},

A {item.type} with title {item.label} has been marked as hidden.
View Reports: {queue.entry}
EOD;


	$defaults['user_user_hidden_message'] = <<<EOD
Hello {user.displayname},

Your account has been reported as abusive and is under moderation.

Please contact our support for further information.
Regards

EOD;
	$defaults['user_user_restored_message'] = <<<EOD
Hello {user.displayname},

Thank you for your patience. Your account has been restored.

Please contact our support if you need further support.
Regards

EOD;

	$defaults['user_user_deleted_message'] = <<<EOD
Hello {user.displayname},

Your account has been removed due to abusive content.

Regards

EOD;


	$defaults['post_admin_reported_message'] = <<<'EOD'
Hello {user.displayname},

A new {item.type} with title {item.label} has been marked as abusive.

Reported Content: {report.content}

Reported By:   {report.reporter}
Reported Date: {report.date}
View Report:   {report.link}

EOD;

	$defaults['post_admin_hidden_message'] = <<<'EOD'
Hello {user.displayname},

A {item.type} with title {item.label} has been marked as hidden.
View Reports: {queue.entry}
EOD;


	$defaults['post_author_hidden_message'] = <<<'EOD'
Hello {user.displayname},

Your post {item.label} has been reported as abusive and is under moderation.

Please contact our support for further information.
Regards
EOD;

	$defaults['post_author_restored_message'] = <<<'EOD'
Hello {user.displayname},

Thank you for your patience. Your post {item.label} has been restored.

Please contact our support if you need further support.
Regards
EOD;

	$defaults['post_author_deleted_message'] = <<<'EOD'
Hello {user.displayname},

Your post {item.label} has been removed due to abusive content.

Regards
EOD;

	// Groups.
	$defaults['bp_group_admin_reported_message'] = <<<'EOD'
Hello {user.displayname},

A new {item.type} with title {item.label} has been marked as abusive.

Reported Content: {report.content}

Reported By:   {report.reporter}
Reported Date: {report.date}
View Report:   {report.link}

EOD;

	$defaults['bp_group_admin_hidden_message'] = <<<'EOD'
Hello {user.displayname},

A {item.type} with title {item.label} has been marked as hidden.
View Reports: {queue.entry}
EOD;


	$defaults['bp_group_gadmin_hidden_message'] = <<<'EOD'
Hello {user.displayname},

Your group {item.label} has been reported as abusive and is under moderation.

Please contact our support for further information.
Regards

EOD;
	$defaults['bp_group_gadmin_restored_message'] = <<<'EOD'
Hello {user.displayname},

Thank you for your patience. Your group {item.label} has been restored.

Please contact our support if you need further support.
Regards

EOD;

	$defaults['bp_group_gadmin_deleted_message'] = <<<'EOD'
Hello {user.displayname},

Your group {item.label} has been removed due to abusive content.

Regards

EOD;


	$defaults['bp_activity_admin_reported_message'] = <<<'EOD'
Hello {user.displayname},

A new {item.type} with content {item.label} has been marked as abusive.

Reported Date: {report.date}
Reported Content: {report.content}
Reported By: {report.reporter}
View Report: {report.link}

EOD;

	$defaults['bp_activity_admin_hidden_message'] = <<<'EOD'
Hello {user.displayname},

A {item.type} with title {item.label} has been marked as hidden.
View Reports: {queue.entry}
EOD;


	$defaults['bp_activity_user_hidden_message'] = <<<'EOD'
Hello {user.displayname},

Your activity {item.label} has been reported as abusive and is under moderation.

Please contact our support for further information.
Regards

EOD;
	$defaults['bp_activity_user_restored_message'] = <<<'EOD'
Hello {user.displayname},

Thank you for your patience. Your activity {item.label} has been restored.

Please contact our support if you need further support.
Regards

EOD;

	$defaults['bp_activity_user_deleted_message'] = <<<'EOD'
Hello {user.displayname},

Your activity {item.label} has been removed due to abusive content.

Regards

EOD;

	// Activity comment.
	$defaults['bp_acomment_admin_reported_message'] = <<<'EOD'
Hello {user.displayname},

A new {item.type} with content {item.label} has been marked as abusive.

Reported Date: {report.date}
Reported Content: {report.content}
Reported By: {report.reporter}
View Report: {report.link}

EOD;

	$defaults['bp_acomment_admin_hidden_message'] = <<<'EOD'
Hello {user.displayname},

A {item.type} with title {item.label} has been marked as hidden.
View Reports: {queue.entry}
EOD;


	$defaults['bp_acomment_user_hidden_message'] = <<<'EOD'
Hello {user.displayname},

Your activity {item.label} has been reported as abusive and is under moderation.

Please contact our support for further information.
Regards

EOD;
	$defaults['bp_acomment_user_restored_message'] = <<<'EOD'
Hello {user.displayname},

Thank you for your patience. Your activity {item.label} has been restored.

Please contact our support if you need further support.
Regards

EOD;

	$defaults['bp_acomment_user_deleted_message'] = <<<'EOD'
Hello {user.displayname},

Your activity {item.label} has been removed due to abusive content.

Regards

EOD;

	$defaults['bbp_reply_admin_reported_message'] = <<<'EOD'
Hello {user.displayname},

A new {item.type} with title {item.label} has been marked as abusive.

Reported Content: {report.content}

Reported By:   {report.reporter}
Reported Date: {report.date}
View Report:   {report.link}

EOD;

	$defaults['bbp_reply_admin_hidden_message'] = <<<'EOD'
Hello {user.displayname},

A {item.type} with title {item.label} has been marked as hidden.
View Reports: {queue.entry}
EOD;


	$defaults['bbp_reply_user_hidden_message'] = <<<'EOD'
Hello {user.displayname},

Your reply {item.label} has been reported as abusive and is under moderation.

Please contact our support for further information.
Regards
EOD;

	$defaults['bbp_reply_user_restored_message'] = <<<'EOD'
Hello {user.displayname},

Thank you for your patience. Your reply {item.label} has been restored.

Please contact our support if you need further support.
Regards
EOD;

	$defaults['bbp_reply_user_deleted_message'] = <<<'EOD'
Hello {user.displayname},

Your reply {item.label} has been removed due to abusive content.

Regards
EOD;

	$defaults['bbp_forum_admin_reported_message'] = <<<'EOD'
Hello {user.displayname},

A new {item.type} with title {item.label} has been marked as abusive.

Reported Content: {report.content}

Reported By:   {report.reporter}
Reported Date: {report.date}
View Report:   {report.link}

EOD;

	$defaults['bbp_forum_admin_hidden_message'] = <<<'EOD'
Hello {user.displayname},

A {item.type} with title {item.label} has been marked as hidden.
View Reports: {queue.entry}
EOD;


	$defaults['bbp_forum_user_hidden_message'] = <<<'EOD'
Hello {user.displayname},

Your forum {item.label} has been reported as abusive and is under moderation.

Please contact our support for further information.
Regards
EOD;

	$defaults['bbp_forum_user_restored_message'] = <<<'EOD'
Hello {user.displayname},

Thank you for your patience. Your forum {item.label} has been restored.

Please contact our support if you need further support.
Regards
EOD;

	$defaults['bbp_forum_user_deleted_message'] = <<<'EOD'
Hello {user.displayname},

Your forum {item.label} has been removed due to abusive content.

Regards
EOD;

	$defaults['bbp_topic_admin_reported_message'] = <<<'EOD'
Hello {user.displayname},

A new {item.type} with title {item.label} has been marked as abusive.

Reported Content: {report.content}

Reported By:   {report.reporter}
Reported Date: {report.date}
View Report:   {report.link}

EOD;

	$defaults['bbp_topic_admin_hidden_message'] = <<<'EOD'
Hello {user.displayname},

A {item.type} with title {item.label} has been marked as hidden.
View Reports: {queue.entry}
EOD;


	$defaults['bbp_topic_user_hidden_message'] = <<<'EOD'
Hello {user.displayname},

Your topic {item.label} has been reported as abusive and is under moderation.

Please contact our support for further information.
Regards
EOD;

	$defaults['bbp_topic_user_restored_message'] = <<<'EOD'
Hello {user.displayname},

Thank you for your patience. Your topic {item.label} has been restored.

Please contact our support if you need further support.
Regards
EOD;

	$defaults['bbp_topic_user_deleted_message'] = <<<'EOD'
Hello {user.displayname},

Your topic {item.label} has been removed due to abusive content.

Regards
EOD;

	return apply_filters( 'bpmts_default_settings', $defaults );
}

/**
 * Get setting value
 *
 * @param string $option_key Setting name.
 * @param mixed  $default default value.
 *
 * @return mixed
 */
function bpmts_get_option( $option_key, $default = null ) {

	$settings = get_option( 'bpmts-settings', bpmts_get_default_settings() );

	if ( isset( $settings[ $option_key ] ) ) {
		return $settings[ $option_key ];
	}

	return $default;
}

/**
 * Check weather user can be reportable or not
 *
 * @param string $item_type Item type.
 * @param int    $item_id Item id.
 *
 * @return bool
 */
function bpmts_is_item_reportable( $item_type, $item_id ) {
	$reportable_items = bpmts_get_option( 'reportable_items', array() );

	return apply_filters( 'bpmts_is_item_reportable', in_array( $item_type, $reportable_items, true ), $item_type, $item_id );
}

/**
 * Check weather user can report or not
 *
 * @return bool
 */
function bpmts_user_can_report() {

	$can = false;
	$who_can_report = bpmts_get_option( 'who_can_report' );

	if ( 'logged' === $who_can_report && is_user_logged_in() ) {
		$can = true;
	} elseif ( 'all' === $who_can_report ) {
		$can = true;
	}

	return apply_filters( 'bpmts_user_can_report', $can );
}


/**
 * Check if user has already reported the queued item.
 *
 * @param int $user_id user id.
 * @param int $queue_id queue id.
 *
 * @return bool
 */
function bpmts_user_has_reported_queued_item( $user_id, $queue_id ) {
	if ( ! $user_id ) {
		return false;
	}

	return Report::exists( array( 'queue_id' => $queue_id, 'reporter_id' => $user_id ) );
}

/**
 * Check if user has already reported the queued item.
 *
 * @param int    $user_id user id.
 * @param int    $item_id item id.
 * @param string $item_type item type.
 *
 * @return bool
 */
function bpmts_user_has_reported_item( $user_id, $item_id, $item_type ) {
	if ( ! $user_id ) {
		return false;
	}

	$queue = Queue::first( array( 'item_type' => $item_type, 'item_id' => $item_id ) );
	if ( ! $queue ) {
		return false;
	}

	return Report::exists( array( 'queue_id' => $queue->id, 'reporter_id' => $user_id ) );
}

/**
 * Get the IP of the User.
 *
 * @return mixed
 */
function bpmts_get_user_ip() {
	$ip = $_SERVER['REMOTE_ADDR'];

	if ( isset( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ) {
		$ips = explode( ',', $_SERVER['HTTP_X_FORWARDED_FOR'] );
		$ip  = array_pop( $ips );
	}

	return $ip;
}

/**
 * Add New report.
 *
 * @param array $args args.
 *
 * @return bool|WP_Error
 */
function bpmts_report_item( $args ) {

	$defaults = array(
		'item_id'    => 0,
		'item_type'  => '',
		'context_id' => 0,
		'context'    => '',
		'subject'    => '',
		'message'    => '',
		'category'   => 0,
		'name'       => '',
		'email'      => '',
		'ip'         => '',
		'user_id'    => get_current_user_id(),
		'fields'     => '',
	);

	$args = wp_parse_args( $args, $defaults );

	$item_id    = $args['item_id'];
	$item_type  = $args['item_type'];
	$context    = $args['context'];
	$context_id = $args['context_id'];

	if ( ! $item_id || ! $item_type ) {
		return new WP_Error( 'invalid_request', __( 'Invalid action!', 'buddypress-moderation-tools' ) );
	}

	$reportable = bpmts_moderation_tools()->reportable_types->get( $item_type );

	// must be an active and valid reportable type.
	if ( ! $reportable || ! $reportable->active || ! $reportable->is_valid( $item_id, $context, $context_id ) ) {
		return new \WP_Error( 'invalid_type', __( 'Invalid action!', 'buddypress-moderation-tools' ) );
	}

	$form_fields     = bpmts_get_option( 'form_fields', array() );
	$required_fields = bpmts_get_option( 'required_form_fields', array() );

	foreach ( $form_fields as $field ) {
		$val = isset( $args[ $field ] ) ? $args[ $field ] : '';
		$err = bpmts_validate_feedback_field( $field, $val, in_array( $field, $required_fields, true ) );
		// break on first error.
		if ( $err ) {
			return new \WP_Error( 'invalid_field', $err );
		}
	}

	if ( ! bpmts_is_item_reportable( $item_type , $item_id ) ) {
		return new \WP_Error( 'invalid_type', __( 'Can not be reported.', 'buddypress-moderation-tools' ) );
	}

	if ( ! bpmts_user_can_report() ) {
		return new \WP_Error( 'permission_error', __( 'You can not report.', 'buddypress-moderation-tools' ) );
	}

	$queue = bpmts_queue_add_item( $args['item_id'], $item_type, $context_id, $context );

	if ( ! $queue ) {
		return new \WP_Error( 'db_error', __( 'There was a problem in reporting. Please try again.', 'buddypress-moderation-tools' ) );
	}

	$queue_id = $queue->id;


	if ( is_user_logged_in() && bpmts_user_has_reported_queued_item( get_current_user_id(), $queue_id ) ) {
		return new \WP_Error( 'duplicate_report', __( 'You already reported it earlier.', 'buddypress-moderation-tools' ) );
	}

	$user  = bpmts_get_current_user();
	$name  = ! empty( $args['name'] ) ? trim( $args['name'] ) : '';
	$email = isset( $args['email'] ) ? trim( $args['email'] ) : '';

	if ( ! $name ) {
		$name = $user->display_name;
	}

	if ( ! $email ) {
		$email = $user->user_email;
	}

	$report_args = array(
		'reporter_id' => $user->ID,
		'queue_id'    => $queue_id,
		'name'        => $name,
		'email'       => $email,
		'ip'          => bpmts_get_user_ip(),
	);

	if ( in_array( 'subject', $form_fields, true ) ) {
		$report_args['subject'] = $args['subject'];

	}

	if ( in_array( 'message', $form_fields, true ) ) {
		$report_args['message'] = $args['message'];
	}

	if ( in_array( 'category', $form_fields, true ) ) {
		$report_args['category'] = $args['category'];
	}

	$report = Report::create( $report_args );
	if ( $report->save() ) {
		$report->refresh(); // reload the model.
		// bpmts_user_reported etc action.
		do_action( "bpmts_{$item_type}_reported", $item_id, $queue, $report );
		return bpmts_queue_update_item_on_report( $queue );
	} else {
		return new WP_Error( 'db_err', __( 'There was a problem. Please try again later.', 'buddypress-moderation-tools' ) );
	}
}

/**
 * Restore an item.
 *
 * @param int    $item_id item id.
 * @param string $item_type item type.
 *
 * @return bool
 */
function bpmts_restore_item( $item_id, $item_type ) {

	$queue = Queue::first( array( 'item_id' => $item_id, 'item_type' => $item_type, 'is_new' => 1, 'is_hidden' => 1 ) );

	if ( is_null( $queue ) ) {
		return false;
	}

	$reportable = bpmts_moderation_tools()->reportable_types->get( $item_type );

	if ( ! $reportable ) {
		return false;
	}

	$keep_log = bpmts_get_option( 'keep_log', 0 );

	if ( $keep_log ) {


		$log = Log::create( array(
			'queue_id'     => $queue->id,
			'moderator_id' => get_current_user_id(),
			'message'      => sprintf( __( '%1$s restored %2$s: <a href="%3$s">%4$s</a>', 'buddypress-moderation-tools' ), bp_get_loggedin_user_fullname(),  $reportable->label, $reportable->get_item_permalink( $item_id ), $reportable->get_item_title( $item_id ) ),
			'action'       => 'restored',
		) );

		if ( ! ( $log_id = $log->save() ) ) {
			return false;
		}

		Report::update( array( 'log_id' => $log_id ), array( 'queue_id' => $queue->id, 'log_id' => 0 ) );
	}

	if ( ! $keep_log ) {
		Report::destroy( array( 'queue_id' => $queue->id ) );
		Log::destroy( array( 'queue_id' => $queue->id ) );
	}

	$queue->is_new                = 0;
	$queue->is_hidden             = 0;
	$queue->current_reports_count = 0;
	$queue->save();

	bpmts_moderation_tools()->cache->delete($item_type . '_'. $item_id );

	do_action( "bpmts_{$queue->item_type}_restored", $queue->item_id, $queue );
	return true;
}

/**
 * Mark an item clean.
 *
 * Marking clean is not restoration.
 * In marking clean we assume that the tem was not hidden as the reports count did not cross the threshold.
 *
 * @param int    $item_id item id.
 * @param string $item_type item type.
 *
 * @return bool
 */
function bpmts_clean_item( $item_id, $item_type ) {

	$queue = Queue::first( array( 'item_id' => $item_id, 'item_type' => $item_type, 'is_new' => 1 ) );
	if ( is_null( $queue ) ) {
		return false;
	}

	$keep_log = bpmts_get_option( 'keep_log', 0 );

	$reportable = bpmts_moderation_tools()->reportable_types->get( $item_type );

	if ( ! $reportable ) {
		return false;
	}

	if ( $keep_log ) {

		$log = Log::create( array(
			'queue_id'     => $queue->id,
			'moderator_id' => get_current_user_id(),
			'message'      => sprintf( __( '%1$s marked %2$s: <a href="%3$s">%4$s</a> clean', 'buddypress-moderation-tools' ), bp_get_loggedin_user_fullname(),  $reportable->label, $reportable->get_item_permalink( $item_id ), $reportable->get_item_title( $item_id ) ),
			'action'       => 'cleaned',
		) );

		if ( ! ( $log_id = $log->save() ) ) {
			return false;
		}

		Report::update( array( 'log_id' => $log_id ), array( 'queue_id' => $queue->id, 'log_id' => 0 ) );
	}

	if ( ! $keep_log ) {
		Report::destroy( array( 'queue_id' => $queue->id ) );
		Log::destroy( array( 'queue_id' => $queue->id ) );
	}

	$queue->is_new                = 0;
	$queue->is_hidden             = 0;
	$queue->current_reports_count = 0;
	$queue->save();


	bpmts_moderation_tools()->cache->delete($item_type . '_'. $item_id );

	do_action( "bpmts_{$queue->item_type}_cleaned", $queue->item_id, $queue );
	return true;
}

/**
 * Delete an item.
 *
 * @param int    $item_id item id.
 * @param string $item_type item type.
 *
 * @return bool
 */
function bpmts_delete_item( $item_id, $item_type ) {
// 'is_hidden' => 1
	$queue = Queue::first( array( 'item_id' => $item_id, 'item_type' => $item_type, 'is_new' => 1, ) );
	if ( is_null( $queue ) ) {
		return false;
	}

	// validate reportable type.
	$reportable = bpmts_moderation_tools()->reportable_types->get( $item_type );

	// must be an active and valid reportable type.
	if ( ! $reportable || ! $reportable->active ) {
		return false;
	}

	$item = $reportable->get_item( $item_id );

	// let the other codes do their magic.
	do_action( "bpmts_before_{$queue->item_type}_delete", $queue->item_id, $item );

	$keep_log = bpmts_get_option( 'keep_log', 0 );

	if ( $keep_log ) {

		$log = Log::create( array(
			'queue_id'     => $queue->id,
			'moderator_id' => get_current_user_id(),
			'message'      => sprintf( __( '%1$s deleted %2$s: <a href="%3$s">%4$s</a>', 'buddypress-moderation-tools' ), bp_get_loggedin_user_fullname(),  $reportable->label, $reportable->get_item_permalink( $item_id ), $reportable->get_item_title( $item_id ) ),
			'action'       => 'deleted',
		) );

		if ( ! ( $log_id = $log->save() ) ) {
			return false;
		}

		Report::update( array( 'log_id' => $log_id ), array( 'queue_id' => $queue->id, 'log_id' => 0 ) );
	}

	// if we do not want to keep the log.
	if ( ! $keep_log ) {
		Report::destroy( array( 'queue_id' => $queue->id ) );
		Log::destroy( array( 'queue_id' => $queue->id ) );
		Queue::destroy( array( 'id' => $queue->id ) );
	}

	// if we are here, simply call the delete item option.
	$reportable->delete( $item_id );

	$queue->is_new = 0;
	$queue->is_hidden = 0;
	$queue->current_reports_count = 0;
	$queue->save();


	bpmts_moderation_tools()->cache->delete($item_type . '_'. $item_id );

	do_action( "bpmts_{$queue->item_type}_deleted", $queue->item_id, $item );

	return true;
}

/**
 * Add/Update que item and return id.
 *
 * @param int    $item_id reported item id.
 * @param string $item_type reported item type.
 * @param int    $context_id context id.
 * @param string $context context type.
 *
 * @return Queue|bool
 */
function bpmts_queue_add_item( $item_id, $item_type, $context_id, $context ) {
	$item = Queue::first( array( 'item_id' => $item_id, 'item_type' => $item_type ) );

	if ( is_null( $item ) ) {
		$item = Queue::create( array(
			'item_id'    => $item_id,
			'item_type'  => $item_type,
			'context_id' => $context_id,
			'context'    => $context,
			'user_id' => 0,
		) );

		$id = $item->save();
		return $id ? $item : false;
	}

	$changed = false;

	if ( $item->context != $context ) {
		$item->context = $context;
		$changed       = true;
	}

	if ( $item->context_id !== $context_id ) {
		$item->context_id = $context_id;
		$changed          = true;
	}

	if ( ! $item->is_new ) {
		$item->is_new = 1;
		$changed      = true;
	}

	if ( $changed ) {
		$item->save();
	}

	return $item;
}

/**
 * Update queue item state.
 *
 * @param Queue $queue queue item.
 *
 * @return bool
 */
function bpmts_queue_update_item_on_report( $queue ) {
	$reportable = bpmts_moderation_tools()->reportable_types->get( $queue->item_type );

	if ( ! $reportable ) {
		return false;
	}

	// should we update the queue report count?
	if ( empty( $queue->total_reports_count ) ) {
		$queue->total_reports_count = 0;
	}

	if ( empty( $queue->current_reports_count ) ) {
		$queue->current_reports_count = 0;
	}

	if ( empty( $queue->is_hidden ) ) {
		$queue->is_hidden = 0;
	}


	$queue->total_reports_count   = $queue->total_reports_count + 1;
	$queue->current_reports_count = $queue->current_reports_count + 1;

	$hidden = false;

	// if queue item is not hidden and reached the threshold.
	if ( ! $queue->is_hidden && $reportable->get_threshold() <= $queue->current_reports_count ) {
		$queue->is_hidden = 1;
		$hidden = true;
	}

	$queue->save();

	if ( $hidden ) {
		// ra_user_hidden
		// ra_post_hidden
		// ra_bp_group_hidden etc.

		bpmts_moderation_tools()->cache->delete($reportable->type . '_'. $queue->item_id );
		do_action( "bpmts_{$reportable->type}_hidden", $queue->item_id, $queue );
	}

	return true;
}

/**
 * Delete item  entryfrom Queue.
 *
 * @param Queue $queue queue item.
 *
 * @return  bool
 */
function bpmts_queue_delete_entry( $queue ) {

	if ( ! $queue ) {
		return false;
	}

	// delete from reports?
	Report::destroy( array( 'queue_id' => $queue->id ) );
	// delete from logs?
	Log::destroy( ( array( 'queue_id' => $queue->id ) ) );
	// delete from queue.
	$queue->delete();

	return true;
}

/**
 * Get all the registered form fields.
 *
 * @return array
 */
function bpmts_get_registered_form_fields() {

	$fields = array(
		'subject' => __( 'Subject', 'buddypress-moderation-tools' ),
		'message' => __( 'Message', 'buddypress-moderation-tools' ),
		'category' => __( 'Category', 'buddypress-moderation-tools' ),
		'name' => __( 'Reporter Name', 'buddypress-moderation-tools' ),
		'email' => __( 'Reporter Email', 'buddypress-moderation-tools' ),
	);

	return apply_filters( 'bpmts_registered_form_fields', $fields );
}

/**
 * Validate and return error message if any.
 *
 * @param string       $field valid field name.
 * @param string|mixed $value valid field value.
 * @param bool         $is_required is the field required.
 *
 * @return string
 */
function bpmts_validate_feedback_field( $field, $value, $is_required ) {

	$fields = bpmts_get_registered_form_fields();

	$value = trim( $value );

	if ( $is_required && empty( $value ) ) {
		return sprintf( __( '%s is required.', 'buddypress-moderation-tools' ), $fields[ $field ] );
	} elseif ( ! $is_required && empty( $value ) ) {
		return '';
	}

	if ( empty( $value ) && ( 'email' === $field || 'name' === $field ) && is_user_logged_in() ) {
		return '';
	}

	$err = '';
	switch ( $field ) {

		case 'subject':
			$err = strlen( $value ) >= 4 ? '' : sprintf( __( '%s must be at least %d letters.', 'buddypress-moderation-tools' ), $fields[ $field ], 4 );
			break;

		case 'message':
			$err = strlen( $value ) >= 10 ? '' : sprintf( __( '%s must be at least %d letters.', 'buddypress-moderation-tools' ), $fields[ $field ], 10 );
			break;

		case 'category':
			// change to Category::Exists.
			$err = absint( $value ) && true ? '' : sprintf( __( '%s must be selected.', 'buddypress-moderation-tools' ), $fields[ $field ] );
			break;


		case 'name':
			$err = ( ! is_user_logged_in() ) && strlen( $value ) < 3 ? sprintf( __( '%s must be at least %d letters.', 'buddypress-moderation-tools' ), $fields[ $field ], 3 ) : '';
			break;

		case 'email':
			$err = is_email( $value ) ? '' : __( 'Please provide a valid email address.', 'buddypress-moderation-tools' );
			break;

	}

	return apply_filters( 'bpmts_form_field_validation_message', $err, $field, $value );
}
/**
 * Get user object
 *
 * @return object|\WP_User
 */
function bpmts_get_current_user() {

	if ( is_user_logged_in() ) {
		return wp_get_current_user();
	}

	$user = new \stdClass();

	$user->ID           = 0;
	$user->display_name = isset( $_POST['name'] ) ? $_POST['name'] : '';
	$user->user_email   = isset( $_POST['email'] ) ? $_POST['email'] : '';

	return $user;
}
/**
 * Do a safe redirect.
 *
 * @param string $url where to redirect.
 */
function bpmts_redirect( $url ) {
	wp_safe_redirect( $url );
	exit( 0 );
}

/**
 * Check if item is hidden.
 *
 * @param int    $item_id item id.
 * @param string $item_type item type.
 * @param bool   $cache_only only check in the cache.
 *
 * @return bool
 */
function bpmts_is_item_hidden( $item_id, $item_type, $cache_only = false ) {
	$cache = bpmts_moderation_tools()->cache;

	$in_cache = $cache->has( $item_type .'_'. $item_id );

	if ( $cache_only || $in_cache ) {
		return $cache->get( $item_type . '_'. $item_id );
	}

	$queue  = Queue::first( array( 'item_id' => $item_id, 'item_type' => $item_type ) );
	$hidden = $queue && $queue->is_hidden ? true : false;
	bpmts_moderation_tools()->cache->set( $item_type . '_' . $item_id, $hidden );
	return $hidden;
}

/**
 * Is activity hidden?
 *
 * @param int  $activity_id activity id.
 * @param bool $cache_only only check in the cache.
 *
 * @return bool
 */
function bpmts_is_activity_hidden( $activity_id , $cache_only = false) {
	return bpmts_is_item_hidden( $activity_id, 'bp_activity', $cache_only );
}
