<?php
/**
 * BuddyPress Moderation Tools:- User Action handler.
 *
 * @package    BuddyPress Moderation Tools
 * @subpackage User
 * @copyright  Copyright (c) 2018, Brajesh Singh
 * @license    https://www.gnu.org/licenses/gpl.html GNU Public License
 * @author     Brajesh Singh
 * @since      1.0.0
 */

namespace PressThemes\ModerationTools\Reportable_Types\User\Handlers;


use PressThemes\ModerationTools\Messages\Message_Builder;
use PressThemes\ModerationTools\Models\Queue;
use PressThemes\ModerationTools\Models\Report;
use PressThemes\ModerationTools\Reportable_Types\User\Messages\User_Message_Factory;

// Exit if files accessed directly over web.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Action handler.
 */
class User_Action_Handler {

	/**
	 * Boot this class.
	 */
	public static function boot() {
		$self = new self();
		$self->setup();
	}

	/**
	 * Setup hooks callbacks
	 */
	public function setup() {
		// login check. moved to User_Gate
		// add_action( 'wp_login', array( $this, 'check_login' ), 10, 2 );

		// notifications.
		add_action( 'bpmts_user_reported', array( $this, 'on_report' ), 10, 3 );
		add_action( 'bpmts_user_hidden', array( $this, 'on_hide' ), 10, 2 );

		add_action( 'bpmts_user_restored', array( $this, 'on_restore' ) );
		// Notify before delete.
		add_action( 'bpmts_before_user_delete', array( $this, 'on_delete' ), 10, 2 );
	}

	/**
	 * Notify admins via email on new report
	 *
	 * @param int    $user_id reported user id.
	 * @param Queue  $queue queue item.
	 * @param Report $report report object.
	 */
	public function on_report( $user_id, $queue, $report ) {

		// Do not notify if the user is a super admin.
		if ( is_super_admin( $user_id ) ) {
			return;
		}

		// Account activated, send email to user.
		if ( bpmts_get_option( 'user_notify_admin_on_report' ) ) {
			$message_builder = User_Message_Factory::build( 'admin-reported' );

			if ( ! $message_builder ) {
				return;
			}

			$message_builder->add_reports_token( $report );
			$this->notify_admin( $user_id,  $message_builder );
		}
	}

	/**
	 * Notify on account being hidden.
	 *
	 * @param int   $user_id hidden account user id.
	 * @param Queue $queue queue entry.
	 */
	public function on_hide( $user_id, $queue ) {

		// even though super admins are not allowed to deactivate their account, let us keep a check.
		if ( is_super_admin( $user_id ) ) {
			return;
		}

		$is_super_admin = is_super_admin();

		// Account hidden, sed email if allowed.
		if ( bpmts_get_option( 'user_notify_user_on_hide' ) ) {
			$message_builder = User_Message_Factory::build( 'user-hidden' );

			if ( ! $message_builder ) {
				return;
			}

			$this->notify_user( $user_id, $message_builder );
		}

		// Account hidden, admin email.
		if ( ! $is_super_admin && bpmts_get_option( 'user_notify_admin_on_hide' ) ) {
			$message_builder = User_Message_Factory::build( 'admin-hidden' );

			if ( ! $message_builder ) {
				return;
			}

			$message_builder->add_token_map( array( '{queue.entry}' => bpmts_admin_get_queue_entry_url( array( 'queue_id' => $queue->id ) ) ) );
			$this->notify_admin( $user_id, $message_builder );
		}

		// Logout user if our settings allows.
		if ( ! bpmts_is_user_manager( $user_id ) && bpmts_get_option( 'user_logout_on_hide' ) ) {
			// destroy all sessions for the deactivated user.
			bpmts_destroy_user_sessions( $user_id );
		}
	}


	/**
	 * Notify user via email on account activation
	 *
	 * @param int $user_id restored account user id.
	 */
	public function on_restore( $user_id ) {

		// Most probably a wrong case, do not notify super admin.
		if ( is_super_admin( $user_id ) ) {
			return;
		}

		// Account activated, send email to user.
		if ( bpmts_get_option( 'user_notify_user_on_restore' ) ) {
			$message_builder = User_Message_Factory::build( 'user-restored' );

			if ( ! $message_builder ) {
				return;
			}
			$this->notify_user( $user_id, $message_builder );
		}
	}

	/**
	 * Notify on account deleten.
	 *
	 * @param int      $user_id hidden account user id.
	 * @param \WP_User $user object.
	 */
	public function on_delete( $user_id, $user ) {

		// Not for admin.
		if ( is_super_admin( $user_id ) ) {
			return;
		}

		// Notify?
		if ( bpmts_get_option( 'user_notify_user_on_delete' ) ) {
			$message_builder = User_Message_Factory::build( 'user-deleted' );

			if ( ! $message_builder ) {
				return;
			}
			$this->notify_user( $user_id, $message_builder );
		}
	}

	/**
	 * Notify admin on new report or hidden.
	 *
	 * @param int             $user_id id of user account whose status has changed.
	 * @param Message_Builder $message_builder message builder.
	 */
	public function notify_admin( $user_id, $message_builder ) {


		$administrators = bpmts_get_administrators();

		$user = get_userdata( $user_id );

		foreach ( $administrators as $administrator ) {
			$message = $message_builder->parse( $administrator, $user );
			if ( ! $message ) {
				continue;
			}

			wp_mail( $administrator->user_email, $message->subject, $message->message );
		}
	}

	/**
	 * Notify user via email on account hidden/restore.
	 *
	 * @param int             $user_id id of user account whose status has changed.
	 * @param Message_Builder $message_builder message builder.
	 */
	public function notify_user( $user_id, $message_builder ) {

		$user    = get_userdata( $user_id );
		$message = $message_builder->parse( $user, $user );
		wp_mail( $user->user_email, $message->subject, $message->message );
	}
}
