<?php
/**
 * Reportable Types Collection
 *
 * @package    ReportAbuse
 * @subpackage Reportable
 * @copyright  Copyright (c) 2018, Brajesh Singh
 * @license    https://www.gnu.org/licenses/gpl.html GNU Public License
 * @author     Brajesh Singh
 * @since      1.0.0
 */

namespace PressThemes\ModerationTools\Reportable_Types;


// No direct access over web.
use PressThemes\ModerationTools\Contracts\Collection;
use PressThemes\ModerationTools\Contracts\Reportable;

if ( ! defined( 'ABSPATH' ) ) {
	exit( 0 );
}

/**
 * Module manager
 *
 * @see bpmts_moderation_tools()->reportables for accessing it.
 */
class Reportable_Item_Types implements Collection {

	/**
	 * Collection of registered modules.
	 *
	 * @var Reportable_Item_Type[]
	 */
	private $items = array();

	/**
	 * Boot Active Items.
	 */
	public function boot() {

		foreach ( $this->items as $item ) {

			if ( ! $item->active ) {
				continue;
			}

			$item->boot();
		}
	}

	/**
	 * Setup items.
	 */
	public function setup() {

		foreach ( $this->items as $item ) {

			if ( ! $item->active ) {
				continue;
			}

			$item->setup();
		}
	}

	/**
	 * Add a module to the registry.
	 *
	 * @param Reportable_Item_Type $reportable Reportable Type.
	 *
	 * @return bool
	 */
	public function add( $reportable ) {
		$added = false;

		if ( $reportable instanceof Reportable_Item_Type ) {
			$this->items[ $reportable->type ] = $reportable;
			$added                            = true;
		}

		return $added;
	}

	/**
	 * Add an array of Reportable Types to registry.
	 *
	 * @param Reportable[] $reportables Reportable Types array.
	 *
	 * @return bool
	 */
	public function add_multiple( array $reportables ) {
		$added = true;

		foreach ( $reportables as $reportable ) {
			if ( ! $this->add( $reportable ) ) {
				$added = false;
			}
		}

		return $added;
	}

	/**
	 * Get the registered module.
	 *
	 * @param string $type module id.
	 *
	 * @return Reportable_Item_Type|null
	 */
	public function get( $type ) {

		if ( ! $this->has( $type ) ) {
			return null;
		}

		return $this->items[ $type ];
	}

	/**
	 * Get all the registered modules array.
	 *
	 * @return Reportable_Item_Type[]
	 */
	public function get_all() {
		return $this->items;
	}

	/**
	 * An array of active reportable types.
	 *
	 * @return Reportable_Item_Type[]
	 */
	public function get_active() {
		return wp_list_filter( $this->items, array( 'active' => 1 ) );
	}

	/**
	 * Do we have the reportable type as registered.
	 *
	 * @param string $reportable_type reportable type.
	 *
	 * @return bool
	 */
	public function has( $reportable_type ) {
		return $reportable_type && isset( $this->items[ $reportable_type ] );
	}
}
