<?php
/**
 * Moderation History Table.
 *
 * @package    BuddyPress Moderation Tools
 * @subpackage Admin
 * @copyright  Copyright (c) 2018, Brajesh Singh
 * @license    https://www.gnu.org/licenses/gpl.html GNU Public License
 * @author     Brajesh Singh
 * @since      1.0.0
 */

namespace PressThemes\ModerationTools\Admin;

use PressThemes\ModerationTools\Models\Log;
use PressThemes\ModerationTools\Models\Queue;

// Do not allow direct access over web.
defined( 'ABSPATH' ) || exit;

/**
 * Class Log_Table
 */
class Log_Table extends \WP_List_Table {
	/**
	 * Queue items.
	 *
	 * @var Queue[]
	 */
	private $queue_items = array();

	/**
	 * Flag vars.
	 *
	 * @var array
	 */
	private $args = array();

	/**
	 * Log_Table constructor.
	 *
	 * @param array $args Array of values.
	 */
	public function __construct( $args = array() ) {
		$this->args = $args;

		$parent_args = array(
			'singular' => 'log_item',
			'plural'   => 'log_items',
			'screen'   => null,
			'ajax'     => false,
		);
		parent::__construct( $parent_args );
	}

	/**
	 * Check user permissions
	 *
	 * @return bool
	 */
	public function ajax_user_can() {
		return current_user_can( 'manage_options' );
	}

	/**
	 * All logic goes here
	 */
	public function prepare_items() {
		$current_page = $this->get_pagenum();
		$per_page     = 20;

		$args = array(
			'per_page' => $per_page,
			'is_new'   => 1,
			'page'     => $current_page,
		);

		if ( ! empty( $this->args['type'] ) ) {
			$args['item_type'] = $this->args['type'];
		}

		if ( ! empty( $this->args['hidden'] ) ) {
			$args['is_hidden'] = $this->args['hidden'];
		}

		$args['orderby'] = 'updated_at';
		$args['order']   = 'DESC';

		$this->items = Log::get( $args );

		$queue_ids = wp_list_pluck( $this->items, 'queue_id' );

		$queue_items = Queue::get( array(
			'id' => array(
				'op'    => 'IN',
				'value' => $queue_ids,
			),
		) );

		$queue_items_organized = array();

		foreach ( $queue_items as $item ) {
			$queue_items_organized[ $item->id ] = $item;
		}

		$this->queue_items = $queue_items_organized;

		unset( $args['per_page'] );
		unset( $args['offset'] );

		$this->set_pagination_args( array(
			'total_items' => Log::count( $args ),
			'per_page'    => $per_page,
		) );

	}

	/**
	 * Render message when no items found
	 */
	public function no_items() {
		_e( 'The log is empty.', 'buddypress-moderation-tools' );
	}

	/**
	 * Get views
	 *
	 * @return array
	 */
	protected function get_views() {
		$class = empty( $this->args['type'] ) ? 'current' : '';

		$views = array( 'all' => sprintf( '<a href="%s" class="%s">%s</a>', bpmts_admin_get_log_url(), $class, __( 'All', 'buddypress-moderation-tools' ) ) );

		return $views;
		/*
		$reportable_items = bpmts_moderation_tools()->reportable_types->get_all();
		foreach ( $reportable_items as $item ) {
			if ( ! $item->active ) {
				continue;
			}

			$class = $item->type === $this->args['type'] ? 'current' : '';

			$views[ 'bpmts-type-' . $item->type ] = sprintf( '<a href="%s" class="%s">%s</a>', bpmts_admin_get_log_url( array( 'bpmts_type' => $item->type ) ), $class, $item->label );
		}

		return $views;
		*/
	}

	/**
	 * Return bulk actions
	 *
	 * @return array
	 */
	protected function get_bulk_actions() {
		$bulk_actions = array(
			'delete' => __( 'Delete', 'buddypress-moderation-tools' ),
		);

		return $bulk_actions;
	}

	/**
	 * Get column info
	 *
	 * @return array
	 */
	public function get_columns() {
		return array(
			'cb'              => '<input type="checkbox" />',
			'thumbnail'       => __( 'Thumbnail', 'buddypress-moderation-tools' ),
			'item'            => __( 'Reported Item', 'buddypress-moderation-tools' ),
			'type'            => __( 'Type', 'buddypress-moderation-tools' ),
			'user_id'         => __( 'Related user', 'buddypress-moderation-tools' ),
			'moderator_id'    => __( 'Moderated By', 'buddypress-moderation-tools' ),
			'current_reports' => __( 'Reports', 'buddypress-moderation-tools' ),
			'updated_at'      => __( 'Last Updated', 'buddypress-moderation-tools' ),
		);
	}

	/**
	 * Sortable columns.
	 *
	 * @return array
	 */
	public function get_sortable_columns() {

		$sortable_columns = array(
			'updated_at' => array( 'updated_at', false ),
		);

		return $sortable_columns;
	}

	/**
	 * Mask action
	 *
	 * @param Log $item Current queue id.
	 *
	 * @return string
	 */
	public function column_cb( $item ) {
		return sprintf(
			'<input type="checkbox" name="items[]" value="%d" />', $item->id
		);
	}

	/**
	 * Column data.
	 *
	 * @param Log    $log log item.
	 * @param string $col column name.
	 *
	 * @return string
	 */
	public function column_default( $log, $col ) {

		$item = $this->queue_items[ $log->queue_id ];

		if ( ! $item ) {
			return 'N/A';
		}

		switch ( $col ) {

			case 'id':
				return $item->id;
			case 'user_id':
				$user = bp_core_fetch_avatar( array( 'item_id' => $item->user_id, 'size' => 25 ) ) .'<br/>'.  bp_core_get_userlink( $item->user_id );
				return $user;

			case 'moderator_id':
				$user = bp_core_fetch_avatar( array( 'item_id' => $log->moderator_id, 'size' => 25 ) ) .'<br/>'.  bp_core_get_userlink( $log->moderator_id );
				return $user;
			case 'total_reports':
				return $item->total_reports_count;

			case 'current_reports':
				return $item->current_reports_count;

			case 'type':
				$reportable = bpmts_moderation_tools()->reportable_types->get( $item->item_type );
				return $reportable ? $reportable->label : $item->item_type;

			case 'updated_at':
				return mysql2date( 'g:i:s A, F j, Y', $item->updated_at );
		}
	}


	/**
	 * Get the item column.
	 *
	 * @param Log $log queue item.
	 *
	 * @return string
	 */
	public function column_item( $log ) {
		$item = $this->queue_items[ $log->queue_id ];

		if ( ! $item ) {
			return 'N/A';
		}
		$reportable = bpmts_moderation_tools()->reportable_types->get( $item->item_type );
		if ( ! $reportable ) {
			return __( 'Unavailable type', 'buddypress-moderation-tools' );
		}

		return sprintf( '<div class="bpmts-title-col-entry bpmts-type-{%s}-title-col-entry">%s</div>',$reportable->type, $reportable->get_item_title( $item->item_id ) );
	}

	/**
	 * Get the item column.
	 *
	 * @param Log $log log item.
	 *
	 * @return string
	 */
	public function column_thumbnail( $log ) {
		$item = $this->queue_items[ $log->queue_id ];

		if ( ! $item ) {
			return 'N/A';
		}

		$reportable = bpmts_moderation_tools()->reportable_types->get( $item->item_type );
		if ( ! $reportable ) {
			return __( 'Unavailable type', 'buddypress-moderation-tools' );
		}

		return sprintf( '<div class="bpmts-thumb-col-entry bpmts-type-{%s}-thumb-col-entry">%s</div>',$reportable->type, $reportable->get_item_thumbnail( $item->item_id, 'thumb' ) );
	}

	/**
	 * Handle row actions.
	 *
	 * @param Log    $item Queue item object.
	 * @param string $column_name column name.
	 * @param string $primary is primary column.
	 *
	 * @return bool|int|string
	 */
	public function handle_row_actions( $item, $column_name, $primary ) {
		$actions = array();

		switch ( $column_name ) {
			case 'item':
			case 'actions':
			case 'thumbnail':
				$url = add_query_arg(
					array(
						'log_id' => $item->id,
						'page'   => 'bpmts-log',
					), admin_url( 'admin.php' )
				);

				$view_url   = add_query_arg( array( 'action' => 'view-reports' ), $url );
				$delete_url = add_query_arg( array( 'action' => 'delete-reports' ), $url );
				$delete_url = wp_nonce_url( $delete_url, 'delete-reports' );
				$actions    = array(
					'view'   => sprintf( '<a href="%s" title="%s">%s</a>', $view_url, __( 'View all reports', 'buddypress-moderation-tools' ), __( 'View', 'buddypress-moderation-tools' ) ),
					'delete' => sprintf( '<a href="%s" title="%s">%s</a>', $delete_url, __( 'Delete all reports', 'buddypress-moderation-tools' ), __( 'Delete', 'buddypress-moderation-tools' ) ),
				);

				break;
		}

		return $this->row_actions( $actions );
	}

}
