<?php
/**
 * Admin Loader.
 *
 * @package    BuddyPress Moderation Tools
 * @subpackage Admin
 * @copyright  Copyright (c) 2018, Brajesh Singh
 * @license    https://www.gnu.org/licenses/gpl.html GNU Public License
 * @author     Brajesh Singh
 * @since      1.0.0
 */

namespace PressThemes\ModerationTools\Admin;

use PressThemes\ModerationTools\Models\Log;
use PressThemes\ModerationTools\Models\Report;

// Do not allow direct access over web.
defined( 'ABSPATH' ) || exit;

/**
 * Log manager
 *
 * @property-read string $bulk_action bulk action.
 * @property-read string $action moderation action.
 * @property-read int $log_id queue id.
 * @property-read int $report_id queue id.
 */
class Log_Manager {

	/**
	 * Args.
	 *
	 * @var array
	 */
	private $args = array();

	/**
	 * List table.
	 *
	 * @var \WP_List_Table
	 */
	private $table = null;

	/**
	 * Queue_Manager constructor.
	 *
	 * @param array $args args.
	 */
	public function __construct( $args = array() ) {
		$this->args = $args;
	}

	/**
	 * Magic getter.
	 *
	 * @param string $name name.
	 *
	 * @return mixed|null
	 */
	public function __isset( $name ) {
		return isset( $this->args[ $name ] ) ? true : false;
	}
	/**
	 * Magic getter.
	 *
	 * @param string $name name.
	 *
	 * @return mixed|null
	 */
	public function __get( $name ) {
		return isset( $this->args[ $name ] ) ? $this->args[ $name ] : null;
	}

	/**
	 * Get Table.
	 *
	 * @return \WP_List_Table
	 */
	public function get_table() {
		return $this->table;
	}

	/**
	 * Process log actions.
	 */
	public function process() {

		if ( empty( $this->log_id ) ) {
			$this->bulk_delete();
			$this->table = new Log_Table( $this->args );
		} elseif ( $this->report_id ) {
			// single report action.
			$this->handle_log_reports_actions();
		} else {
			// view queue entry log.
			$this->handle_log_entry_actions();
		}
	}

	/**
	 * Handle queue reports for single report.
	 */
	private function handle_log_reports_actions() {
		switch ( $this->action ) {

			case 'delete-report':
				$this->delete_report();
				break;
		}
	}

	/**
	 * Handle queue entry actions.
	 */
	private function handle_log_entry_actions() {

		switch ( $this->action ) {

			case 'delete-entry':
				$this->delete_log();
				break;

			case 'view-entry':
			default:
				$this->table = new Reports_Table( $this->args );
				break;
		}
	}

	/**
	 * Bulk Delete Logs.
	 */
	private function bulk_delete() {

		$bulk_action = $this->bulk_action;
		$log_ids     = isset( $_POST['items'] ) ? $_POST['items'] : array();
		if ( empty( $log_ids ) ) {
			return;
		}

		switch ( $bulk_action ) {

			case 'delete':
				// Delete all reports for these logs.
				Report::destroy( array(
					'log_id' => array(
						'op'    => 'IN',
						'value' => $log_ids,
					),
				) );

				Log::destroy( array(
					'id' => array(
						'op'    => 'IN',
						'value' => $log_ids,
					),
				) );

				$args = array(
					'message' => __( 'Deleted.', 'buddypress-moderation-tools' ),
				);
				bpmts_redirect( bpmts_admin_get_log_url( $args ) );
				break;
		}
	}

	/**
	 * Delete Log.
	 */
	private function delete_log() {
		Report::destroy( array( 'log_id' => $this->log_id ) );
		Log::destroy( array( 'id' => $this->log_id ) );
		$message = __( 'Item removed from log.', 'buddypress-moderation-tools' );
		// redirect to the queue.
		bpmts_redirect( bpmts_admin_get_log_url( array( 'message' => $message ) ) );
	}

	/**
	 * Delete report for the log.
	 */
	private function delete_report() {
		Report::destroy( array( 'id' => $this->report_id ) );
		$message = __( 'Report deleted', 'buddypress-moderation-tools' );

		bpmts_redirect( bpmts_admin_get_log_entry_url( array(
			'log_id'  => $this->log_id,
			'message' => $message,
		) ) );

	}
}
