<?php
/**
 * Admin Loader.
 *
 * @package    BuddyPress Moderation Tools
 * @subpackage Admin
 * @copyright  Copyright (c) 2018, Brajesh Singh
 * @license    https://www.gnu.org/licenses/gpl.html GNU Public License
 * @author     Brajesh Singh
 * @since      1.0.0
 */

namespace PressThemes\ModerationTools\Admin;

use PressThemes\ModerationTools\Models\Log;
use PressThemes\ModerationTools\Models\Queue;
use PressThemes\ModerationTools\Models\Report;

// Do not allow direct access over web.
defined( 'ABSPATH' ) || exit;

/**
 * Class Admin_Loader
 */
class Admin_Loader {

	/**
	 * Slug for settings page.
	 *
	 * @var string
	 */
	private $settings_slug = 'bpmts-settings';

	/**
	 * Table object.
	 *
	 * @var \WP_List_Table
	 */
	private $table = null;

	/**
	 * Settings class object
	 *
	 * @var Admin_Settings
	 */

	private $settings = null;

	/**
	 * Queue entry id.
	 *
	 * @var int
	 */
	private $queue_id = 0;

	/**
	 * Queue entry id.
	 *
	 * @var int
	 */
	private $log_id = 0;

	/**
	 * Currently actionable report id.
	 *
	 * @var int
	 */
	private $report_id = 0;


	/**
	 * Current action.
	 *
	 * @var string
	 */
	private $action = '';

	/**
	 * Bulk Action.
	 *
	 * @var string
	 */
	private $bulk_action = '';

	/**
	 * Current type filter.
	 *
	 * @var string
	 */
	private $type = '';

	/**
	 * Current state filter(all| hidden).
	 *
	 * @var int
	 */
	private $hidden = 0;

	/**
	 * Message/Notice.
	 *
	 * @var string
	 */
	private $message = '';

	/**
	 * Page slug.
	 *
	 * @var string
	 */
	private $page_slug = '';

	/**
	 * BPMTS_Admin_Loader constructor.
	 */
	public function __construct() {
	}

	/**
	 * Boot this loader.
	 */
	public static function boot() {
		$self = new self();
		$self->setup();
	}

	/**
	 * Setup callbacks on necessary admin hooks
	 */
	public function setup() {
		add_action( 'admin_menu', array( $this, 'register_menu' ) );
		add_action( 'admin_init', array( $this, 'init' ) );
		add_filter( 'plugin_action_links_' . bpmts_moderation_tools()->basename, array( $this, 'plugin_action_links' ) );
	}

	/**
	 * Register new menu item with subpage
	 */
	public function register_menu() {
		add_menu_page( __( 'Reported Items', 'buddypress-moderation-tools' ), __( 'Moderation', 'buddypress-moderation-tools' ), 'manage_options', 'bpmts-moderation', array(
			$this,
			'render',
		), 'dashicons-flag', 76 );
		add_submenu_page( 'bpmts-moderation', __( 'Awaiting Moderation', 'buddypress-moderation-tools' ), __( 'Queue', 'buddypress-moderation-tools' ), 'manage_options', 'bpmts-moderation', array(
			$this,
			'render',
		) );
		add_submenu_page( 'bpmts-moderation', __( 'History', 'buddypress-moderation-tools' ), __( 'History', 'buddypress-moderation-tools' ), 'manage_options', 'bpmts-log', array(
			$this,
			'render',
		) );
		add_submenu_page( 'bpmts-moderation', __( 'Categories', 'buddypress-moderation-tools' ), __( 'Categories', 'buddypress-moderation-tools' ), 'manage_options', 'bpmts-category', array(
			$this,
			'render',
		) );
		add_submenu_page( 'bpmts-moderation', __( 'Moderation Settings', 'buddypress-moderation-tools' ), __( 'Settings', 'buddypress-moderation-tools' ), 'manage_options', $this->settings_slug, array(
			$this,
			'render',
		) );
	}

	/**
	 * Initialize settings
	 */
	public function init() {

		if ( $this->is_options_page() || $this->is_settings_page() ) {
			$this->settings = new Admin_Settings();
			$this->settings->init();
		}

		$this->handle();
	}

	/**
	 * Add links to plugin entry in plugin list.
	 *
	 * @param array $actions actions.
	 *
	 * @return array
	 */
	public function plugin_action_links( $actions ) {
		if ( is_super_admin() ) {
			$actions['view-bpmts-settings'] = sprintf( '<a href="%1$s" title="%2$s">%3$s</a>', admin_url( 'admin.php?page=bpmts-settings' ), __( 'Settings', 'buddypress-moderation-tools' ), __( 'Settings', 'buddypress-moderation-tools' ) );
		}

		$actions['view-bpmts-docs'] = sprintf( '<a href="%1$s" title="%2$s" target="_blank">%2$s</a>', 'https://buddydev.com/docs/buddypress-moderation-tools/getting-started-with-administration-buddypress-moderation/', __( 'Documentation', 'buddypress-moderation-tools' ) );

		return $actions;
	}

	/**
	 * Setup.
	 */
	private function handle() {
		// parse query string.
		$this->parse_vars();

		$slug = $this->page_slug;

		switch ( $slug ) {

			case 'bpmts-moderation':
			default:
				$this->process_queue_actions();
				break;

			case 'bpmts-log':
				$this->process_log_actions();
				break;

			case 'bpmts-category':
				$this->process_category_actions();
				break;
		}
	}

	/**
	 * Handle screens for Report abuse.
	 */
	private function process_queue_actions() {

		$queue_manager = new Queue_Manager( $this->get_vars() );
		$queue_manager->process();

		$table = $queue_manager->get_table();
		if ( $table ) {
			$this->table = $table;
		}
	}

	/**
	 * Handle screens for Report abuse.
	 */
	private function process_log_actions() {
		$log_manager = new Log_Manager( $this->get_vars() );
		$log_manager->process();
		$table = $log_manager->get_table();

		if ( $table ) {
			$this->table = $table;
		}
	}

	/**
	 * Process category actions.
	 */
	private function process_category_actions() {
		$cat_manager = new Category_Manager();
		$cat_manager->handle();
	}

	/**
	 * Render.
	 */
	public function render() {

		$slug = $this->page_slug;

		switch ( $slug ) {

			case 'bpmts-moderation':
				if ( ! $this->queue_id ) {
					$this->render_queue();
				} else {
					$this->render_queue_entry_reports();
				}
				break;

			default:
				$this->render_queue();
				break;

			case 'bpmts-log':
				if ( ! $this->log_id ) {
					$this->render_log();
				} else {
					$this->render_log_entry_reports();
				}
				break;

			case 'bpmts-settings':
				if ( ! $this->settings ) {
					$this->settings = new Admin_Settings();
				}
				$this->settings->render();
				break;


			case 'bpmts-category':
				$cat_manager = new Category_Manager();
				$cat_manager->render();
				break;
		}
	}

	/**
	 * Render queue screen.
	 */
	private function render_queue() {

		if ( ! $this->table ) {
			wp_die( __( 'Invalid action.', 'buddypress-moderation-tools' ) );
		}

		require dirname( __FILE__ ) . '/templates/queue/queue.php';
	}

	/**
	 * Display reports for an item in the queue.
	 */
	private function render_queue_entry_reports() {
		$entry = Queue::first( array( 'id' => $this->queue_id ) );
		if ( ! $entry || ! $this->table ) {
			wp_die( __( 'Invalid action.', 'buddypress-moderation-tools' ) );
		}

		$reportable = bpmts_moderation_tools()->reportable_types->get( $entry->item_type );
		if ( ! $reportable ) {
			wp_die( __( 'Invalid action.', 'buddypress-moderation-tools' ) );
		}

		require dirname( __FILE__ ) . '/templates/queue/entry-single.php';
	}


	/**
	 * Render logs page.
	 */
	private function render_log() {
		if ( ! $this->table ) {
			wp_die( __( 'Invalid action.', 'buddypress-moderation-tools' ) );
		}
		require dirname( __FILE__ ) . '/templates/log/log.php';
	}

	/**
	 * Display reports for an item in the queue.
	 */
	private function render_log_entry_reports() {
		$entry = Log::first( array( 'id' => $this->log_id ) );
		if ( ! $entry || ! $this->table ) {
			wp_die( __( 'Invalid action.', 'buddypress-moderation-tools' ) );
		}

		$queue      = Queue::find( $entry->queue_id );
		$reportable = bpmts_moderation_tools()->reportable_types->get( $queue->item_type );
		if ( $reportable ) {
			require dirname( __FILE__ ) . '/templates/log/entry-single.php';
		} else {
			echo __( 'Reportable type needs to be active to view the logs.', 'buddypress-moderation-tools' );
		}
	}

	/**
	 * Is the admin settings page for us?
	 *
	 * @return bool
	 */
	private function is_settings_page() {

		if ( isset( $_GET['page'] ) && $this->settings_slug === $_GET['page'] ) {
			return true;
		}

		return false;
	}

	/**
	 * Is it the options.php page that saves settings?
	 *
	 * @return bool
	 */
	private function is_options_page() {
		global $pagenow;

		// We need to load on options.php otherwise settings won't be registered.
		if ( 'options.php' === $pagenow ) {
			return true;
		}

		return false;
	}

	/**
	 * Parse various vars.
	 */
	private function parse_vars() {
		// first check for bulk action.
		$action1 = isset( $_POST['action'] ) && $_POST['action'] != - 1 ? trim( $_POST['action'] ) : '';
		$action2 = isset( $_POST['action2'] ) && $_POST['action2'] != - 1 ? trim( $_POST['action2'] ) : '';

		$this->bulk_action = $action1 ? $action1 : $action2;
		$this->page_slug   = isset( $_GET['page'] ) ? trim( $_GET['page'] ) : '';
		$this->queue_id    = isset( $_GET['queue_id'] ) ? absint( $_GET['queue_id'] ) : 0;
		$this->log_id      = isset( $_GET['log_id'] ) ? absint( $_GET['log_id'] ) : 0;
		$this->report_id   = isset( $_GET['report_id'] ) ? absint( $_GET['report_id'] ) : 0;
		$this->action      = isset( $_GET['bpmts_action'] ) ? trim( $_GET['bpmts_action'] ) : '';
		$this->type        = isset( $_GET['bpmts_type'] ) ? trim( $_GET['bpmts_type'] ) : '';
		$this->hidden      = empty( $_GET['bpmts_state'] ) ? 0 : 1;

		$this->message = isset( $_GET['message'] ) ? $_GET['message'] : '';
	}

	/**
	 * Get current state vars.
	 *
	 * @return array
	 */
	private function get_vars() {
		return array(
			'page'        => $this->page_slug,
			'queue_id'    => $this->queue_id,
			'log_id'      => $this->log_id,
			'report_id'   => $this->report_id,
			'action'      => $this->action,
			'bulk_action' => $this->bulk_action,
			'type'        => $this->type,
			'hidden'      => $this->hidden,
		);
	}

	/**
	 * Used in templates.
	 */
	private function hidden_fields() {
	    ?>
    <input type="hidden" name="page" value="<?php echo esc_attr( $this->page_slug );?>" />
    <?php
    }
}
