<?php
/**
 * BuddyPress Moderation Tools:- User Query filter.
 *
 * @package    BuddyPress Moderation Tools
 * @subpackage User
 * @copyright  Copyright (c) 2018, Brajesh Singh
 * @license    https://www.gnu.org/licenses/gpl.html GNU Public License
 * @author     Brajesh Singh
 * @since      1.0.0
 */

namespace PressThemes\ModerationTools\Reportable_Types\User\Filters;

use PressThemes\ModerationTools\Filters\Query_Filter;

// Do not allow direct access over web.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Filter User Query.
 */
class User_Query_Filter extends Query_Filter {

	/**
	 * Boot this class.
	 */
	public static function boot() {
		$self = new static();
		$self->setup();
	}

	/**
	 * Setup hooks callbacks
	 */
	public function setup() {

		$this->item_type = 'user';

		$this->settings = array(
			'auto_hide' => bpmts_get_option( 'user_auto_hide', 0 ),
			'hide_type' => bpmts_get_option( 'user_hide_type', array( 'all' ) ),
		);

		// hide from members list everywhere.
		add_action( 'pre_user_query', array( $this, 'exclude_in_members_listing' ), 200 );

		// filter total member count.
		add_filter( 'bp_core_get_active_member_count', array( $this, 'fix_active_members_count' ) );

		add_action( 'bp_user_query_populate_extras', array( $this, 'prefetch' ) );


	}

	/**
	 * Pre-fetch member type data when initializing a Members loop.
	 *
	 * @param \BP_User_Query $bp_user_query BP_User_Query object.
	 */
	function prefetch( \BP_User_Query $bp_user_query ) {
		$user_ids = $bp_user_query->user_ids;
		if ( empty ( $user_ids ) ) {
			$user_ids = array();
		}

		if ( is_user_logged_in() ) {
			$user_ids[] = get_current_user_id();
		}

		if ( empty( $user_ids ) ) {
			return;
		}

		$this->cache_hidden_items( $user_ids );
	}

	/**
	 * Exclude users from query.
	 *
	 * @param \WP_User_Query $query query object.
	 */
	public function exclude_in_members_listing( $query ) {
		// do not hide users inside the admin.
		if ( is_super_admin() || ( is_admin() && ! defined( 'DOING_AJAX' ) ) ) {
			return;
		}

		global $wpdb;

		if ( $this->hide_for_reporters_enabled() ) {
			$hidden_reporters_sql = $this->get_sql_for_reported_items( get_current_user_id(), 'user' );
			$query->query_where   .= " AND {$wpdb->users}.ID NOT IN ({$hidden_reporters_sql}) ";
		}

		if ( $this->hide_for_all_enabled() ) {
			$hidden_all_sql     = $this->get_sql_for_hidden_items( 'user' );
			$query->query_where .= " AND {$wpdb->users}.ID NOT IN ({$hidden_all_sql}) ";
		}

	}

	/**
	 * Filter total active members count and subtract the deactivated users count
	 *
	 * @param int $count count.
	 *
	 * @return int
	 */
	public function fix_active_members_count( $count ) {

		// do not hide users inside the admin.
		if ( is_super_admin() || ( is_admin() && ! defined( 'DOING_AJAX' ) ) ) {
			return $count;
		}

		return absint( $count - $this->get_hidden_users_count() );
	}


	/**
	 * Get total no. of hidden users.
	 *
	 * @return int
	 */
	public function get_hidden_users_count() {

		global $wpdb;

		$where_conditions = array();
		if ( $this->hide_for_reporters_enabled() ) {
			$hidden_reporters_sql = $this->get_sql_for_reported_items( get_current_user_id(), 'user' );
			$where_conditions[]   = " {$wpdb->users}.ID IN ({$hidden_reporters_sql}) ";
		}

		if ( $this->hide_for_all_enabled() ) {
			$hidden_all_sql     = $this->get_sql_for_hidden_items( 'user' );
			$where_conditions[] = "{$wpdb->users}.ID IN ({$hidden_all_sql}) ";
		}

		if ( empty( $where_conditions ) ) {
			return 0;
		}

		$where_sql = join( ' OR ', $where_conditions );
		$query     = "SELECT COUNT(ID) FROM {$wpdb->users} WHERE {$where_sql} ";

		return absint( $wpdb->get_var( $query ) );
	}
}
