<?php
/**
 * ReportAbuse:- Post Type filter.
 *
 * @package    ModerationTools
 * @subpackage Reportable_Types\Comment_Type\Filters
 * @copyright  Copyright (c) 2018, Brajesh Singh
 * @license    https://www.gnu.org/licenses/gpl.html GNU Public License
 * @author     Brajesh Singh
 * @since      1.0.0
 */

namespace PressThemes\ModerationTools\Reportable_Types\Comment_Type\Filters;

use PressThemes\ModerationTools\Schema\Schema;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Filter Comment type.
 */
class Comment_Type_Query_Filter {

	/**
	 * Settings.
	 *
	 * @var array
	 */
	private $settings = array();

	/**
	 * Comment type.
	 *
	 * @var string
	 */
	private $comment_type;

	/**
	 * Boot this class.
	 *
	 * @param string $comment_type post type.
	 */
	public static function boot( $comment_type ) {
		$self = new self();
		$self->comment_type = $comment_type;
		$self->setup();
	}

	/**
	 * Setup hooks callbacks
	 */
	public function setup() {
		$this->settings = array(
			'auto_hide' => bpmts_get_option( "{$this->comment_type}_auto_hide", 0 ),
			'hide_type' => bpmts_get_option( "{$this->comment_type}_hide_type", 'all' ),
		);

		// hide posts from list everywhere.
		add_action( 'comments_clauses', array( $this, 'exclude_in_listing' ), 200, 2 );
	}

	/**
	 * Exclude groups from query.
	 *
	 * @param array             $pieces A compacted array of comment query clauses.
	 * @param \WP_Comment_Query $query Current instance of WP_Comment_Query (passed by reference).
	 *
	 * @return string
	 */
	public function exclude_in_listing( $piece, $query ) {
		// do not hide users inside the admin.
		if ( is_super_admin() || ( is_admin() && ! defined( 'DOING_AJAX' ) ) ) {
			return $query;
		}

		// if filter is not needed.
		if ( ! $this->needs_filter() ) {
			return $query;
		}

		if ( 'reporter' === $this->settings['hide_type'] ) {
			$user_id = get_current_user_id();
		} else {
			$user_id = 0;
		}

		$filter_sql = $this->get_hidden_sql( $user_id );

		/**
		 * @todo fix querey
		 */
		if ( ! empty( $filter_sql ) ) {
			$query['WHERE'] .= " AND ID NOT IN($filter_sql)";
		}

		return $query;
	}

	/**
	 * Get exclude post sql.
	 *
	 * @param int $user_id reporter id.
	 *
	 * @return string
	 */
	private function get_hidden_sql( $user_id = 0 ) {
		global $wpdb;
		// I am using a real bad table alias here to avoid any future conflict.
		$table = Schema::table( 'queue' );
		// get all hidden posts.
		$users = $wpdb->prepare( "SELECT raqu.item_id FROM {$table} as raqu WHERE raqu.item_type=%s AND raqu.is_hidden = %d", $this->comment_type, 1 );

		$reports_table = Schema::table( 'reports' );

		if ( $user_id ) {
			$users .= $wpdb->prepare( " AND raqu.id IN( SELECT queue_id FROM {$reports_table} WHERE reporter_id = %d)", $user_id );
		}

		return $users;
	}

	/**
	 * Do we need to filter?
	 *
	 * @return bool
	 */
	private function needs_filter() {

		if ( ! $this->settings['auto_hide'] || is_super_admin() ) {
			return false;
		}

		if ( 'reporter' === $this->settings['hide_type'] && ! is_user_logged_in() ) {
			return false;
		}

		return true;
	}
}
