<?php
/**
 * Reportable support for Comment.
 *
 * @package    ReportAbuse
 * @subpackage Reportable
 * @copyright  Copyright (c) 2018, Brajesh Singh
 * @license    https://www.gnu.org/licenses/gpl.html GNU Public License
 * @author     Brajesh Singh
 * @since      1.0.0
 */

namespace PressThemes\ModerationTools\Reportable_Types\Comment_Type;

use PressThemes\ModerationTools\Reportable_Types\Comment_Type\Handlers\Comment_Type_UI_Helper;
use PressThemes\ModerationTools\Reportable_Types\Reportable_Item_Type;
use PressThemes\ModerationTools\Reportable_Types\Comment_Type\Handlers\Comment_Type_Action_Handler;
use PressThemes\ModerationTools\Reportable_Types\Comment_Type\Filters\Comment_Type_Query_Filter;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Reportable comment type
 */
class Reportable_Comment_Type extends Reportable_Item_Type {

	/**
	 * Comment type
	 *
	 * @var string
	 */
	public $type = '';

	/**
	 * Comment label
	 *
	 * @var string
	 */
	public $label = '';

	/**
	 * Reportable_Comment_Type constructor.
	 *
	 * @param string $comment_type comment type.
	 * @param string $label comment type label.
	 */
	public function __construct( $comment_type, $label ) {

		$this->type = $comment_type;
		$this->label = $label;

		parent::__construct( $comment_type, $label );
	}

	/**
	 * Setup hooks. Called after boot.
	 */
	public function setup() {
		// do some setup.
		// Boot action handlers.
		Comment_Type_Action_Handler::boot( $this->type, $this->label );
		Comment_Type_Query_Filter::boot( $this->type );
		// Boot UI Helper to add messages/buttons as needed.
		Comment_Type_UI_Helper::boot( $this->type );
	}

	/**
	 * Add admin settings.
	 *
	 * @param \Press_Themes\PT_Settings\Page $page page object.
	 */
	public function add_admin_settings( $page ) {
		$panel = $page->add_panel( 'comment-type-' . $this->type, $this->label );
		Comment_Type_Settings_Panel::build( $panel, $this->type, $this->label );
	}

	/**
	 * Check that comment is valid.
	 *
	 * @param int    $item_id Comment id..
	 * @param string $context ignore.
	 * @param int    $context_id ignore.
	 *
	 * @return bool
	 */
	public function is_valid( $item_id, $context, $context_id ) {
		$comment_type = get_comment_type( $item_id );
		return $comment_type && in_array( $comment_type, array( 'comment', 'trackback', 'pingback' ) );
	}


	/**
	 * Delete an item of this type.
	 *
	 * @param int $item_id item id.
	 */
	public function delete( $item_id ) {
		wp_delete_comment( $item_id, true );
	}

	/**
	 * Get the item object.
	 *
	 * @param int $item_id user id.
	 *
	 * @return false|\WP_Comment
	 */
	public function get_item( $item_id ) {
		return get_comment( $item_id );
	}

	/**
	 * Get the thumbnail
	 *
	 * @param int    $item_id item id.
	 * @param string $size thumb|full.
	 *
	 * @return string
	 */
	public function get_item_thumbnail( $item_id, $size ) {
		return '';
	}

	/**
	 * Get item title.
	 *
	 * @param int $item_id item id.
	 *
	 * @return string
	 */
	public function get_item_title( $item_id ) {
		return '';
	}

	/**
	 * Get post description.
	 *
	 * @param int $item_id item id.
	 *
	 * @return string
	 */
	public function get_item_description( $item_id ) {
		return $this->get_item( $item_id )->comment_content;
	}

	/**
	 * Get Post Permalink.
	 *
	 * @param int $item_id user id.
	 *
	 * @return false|string
	 */
	public function get_item_permalink( $item_id ) {
		return get_comment_link( $item_id );
	}

	/**
	 * Get item edit url.
	 *
	 * @param int $item_id item id.
	 *
	 * @return string
	 */
	public function get_item_edit_url( $item_id ) {
		return get_edit_comment_link( $this->get_item( $item_id ) );
	}

	/**
	 * Get item delete url.
	 *
	 * @param int $item_id item id.
	 *
	 * @return string
	 */
	public function get_item_delete_url( $item_id ) {
		return '';
	}
}
