<?php
/**
 * ReportAbuse:- Activity Comment Action handler.
 *
 * @package    ReportAbuse
 * @subpackage BuddyPress/Activity
 * @copyright  Copyright (c) 2018, Brajesh Singh
 * @license    https://www.gnu.org/licenses/gpl.html GNU Public License
 * @author     Brajesh Singh
 * @since      1.0.0
 */

namespace PressThemes\ModerationTools\Reportable_Types\BuddyPress\Activity_Comment\Handlers;

// Exit if files access directly over web.
use PressThemes\ModerationTools\Models\Queue;
use PressThemes\ModerationTools\Models\Report;
use PressThemes\ModerationTools\Reportable_Types\BuddyPress\Activity_Comment\Messages\Activity_Comment_Message_Builder;
use PressThemes\ModerationTools\Reportable_Types\BuddyPress\Activity_Comment\Messages\Activity_Comment_Message_Factory;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Action handler.
 */
class Activity_Comment_Action_Handler {

	/**
	 * Boot this class.
	 */
	public static function boot() {
		$self = new self();
		$self->setup();
	}

	/**
	 * Setup hooks callbacks
	 */
	public function setup() {

		add_action( 'bpmts_bp_acomment_reported', array( $this, 'on_report' ), 10, 3 );
		add_action( 'bpmts_bp_acomment_restored', array( $this, 'on_restore' ) );
		add_action( 'bpmts_bp_acomment_hidden', array( $this, 'on_hide' ), 10, 2 );
		add_action( 'bpmts_before_bp_acomment_delete', array( $this, 'on_delete' ), 10, 2 );
	}

	/**
	 * Notify admins via email on new report
	 *
	 * @param int    $activity_id activity id.
	 * @param Queue  $queue queue entry.
	 * @param Report $report report object.
	 */
	public function on_report( $activity_id, $queue, $report ) {

		// Account activated, send email to user.
		if ( bpmts_get_option( 'bp_acomment_notify_admin_on_report' ) ) {
			$message_builder = Activity_Comment_Message_Factory::build( 'admin-reported' );

			if ( ! $message_builder ) {
				return;
			}

			$message_builder->add_reports_token( $report );
			$this->notify_admin( $activity_id, $message_builder );
		}
	}

	/**
	 * Notify user via email on account being hidden.
	 *
	 * @param int   $activity_id id of group whose status has changed.
	 * @param Queue $queue queue entry.
	 */
	public function on_hide( $activity_id, $queue ) {

		$is_super_admin = is_super_admin();

		// Account deactivated, sed email if allowed.
		if ( bpmts_get_option( 'bp_acomment_notify_user_on_hide' ) ) {
			$message_builder = Activity_Comment_Message_Factory::build( 'user-hidden' );

			if ( ! $message_builder ) {
				return;
			}

			$this->notify_user( $activity_id, $message_builder );
		}

		// Account deactivated by user.
		if ( ! $is_super_admin && bpmts_get_option( 'bp_acomment_notify_admin_on_hide' ) ) {
			$message_builder = Activity_Comment_Message_Factory::build( 'admin-hidden' );

			if ( ! $message_builder ) {
				return;
			}

			$message_builder->add_token_map( array( '{queue.entry}' => bpmts_admin_get_queue_entry_url( array( 'queue_id' => $queue->id ) ) ) );
			$this->notify_admin( $activity_id, $message_builder );
		}
	}

	/**
	 * Notify user via email on account being hidden.
	 *
	 * @param int                   $activity_id id of group whose status has changed.
	 * @param \BP_Activity_Activity $activity activity object.
	 */
	public function on_delete( $activity_id, $activity ) {

		// Account deactivated, sed email if allowed.
		if ( bpmts_get_option( 'bp_acomment_notify_user_on_delete' ) ) {
			$message_builder = Activity_Comment_Message_Factory::build( 'user-deleted' );

			if ( ! $message_builder ) {
				return;
			}
			$this->notify_user( $activity_id, $message_builder );
		}
	}
	/**
	 * Notify user via email on account activation
	 *
	 * @param int $activity_id id of group whose status has changed.
	 */
	public function on_restore( $activity_id ) {

		// Account activated, send email to user.
		if ( bpmts_get_option( 'bp_acomment_notify_user_on_restore' ) ) {
			$message_builder = Activity_Comment_Message_Factory::build( 'user-restored' );

			if ( ! $message_builder ) {
				return;
			}
			$this->notify_user( $activity_id, $message_builder );
		}
	}

	/**
	 * Notify admin on new report or hidden.
	 *
	 * @param int                              $activity_id id of group whose status has changed.
	 * @param Activity_Comment_Message_Builder $message_builder account_activation or account_deactivate.
	 */
	public function notify_admin( $activity_id, $message_builder ) {

		$activity          = new \BP_Activity_Activity( $activity_id );
		$administrators = get_users( array(
			'role' => 'administrator',
		) );


		foreach ( $administrators as $administrator ) {
			$message = $message_builder->parse( $administrator, $activity );
			if ( ! $message ) {
				continue;
			}

			wp_mail( $administrator->user_email, $message->subject, $message->message );
		}
	}

	/**
	 * Notify user via email on activity hidden/restore.
	 *
	 * @param int                              $activity_id id of group whose status has changed.
	 * @param Activity_Comment_Message_Builder $message_builder account_activation or account_deactivate.
	 */
	public function notify_user( $activity_id, $message_builder ) {

		$activity = new \BP_Activity_Activity( $activity_id );


		$user = get_userdata( $activity->user_id );
		if ( ! $user ) {
			return;
		}
		$message = $message_builder->parse( $user, $activity );
		wp_mail( $user->user_email, $message->subject, $message->message );

	}

}
